// WzSLinker 后台服务脚本

class WzSLinkerBackground {
  constructor() {
    this.init();
  }

  init() {
    this.setupInstallListener();
    this.setupMessageListener();
    this.setupActionListener();
    this.setupNotificationListener();
    this.setupUpdateChecker();
    this.setupKeepalive();
    this.setupHealthCheck();
    this.initializeDefaultData();
  }

  // 设置安装监听器
  setupInstallListener() {
    chrome.runtime.onInstalled.addListener(async (details) => {
      if (details.reason === 'install') {
        await this.initializeDefaultData();
        console.log('WzSLinker: 扩展已安装，默认数据已初始化');
      } else if (details.reason === 'update') {
        console.log('WzSLinker: 扩展已更新');
        await this.showUpdateNotification(details.previousVersion);
      }
    });
  }

  // 显示更新通知
  async showUpdateNotification(previousVersion) {
    try {
      const currentVersion = chrome.runtime.getManifest().version;
      
      // 检查是否已经显示过这个版本的更新通知
      const result = await chrome.storage.local.get(['lastUpdateNotification']);
      if (result.lastUpdateNotification === currentVersion) {
        return;
      }

      // 显示更新通知
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icon.png',
        title: 'WzSLinker 已更新',
        message: `从版本 ${previousVersion} 更新到 ${currentVersion}。点击查看更新内容。`
      });

      // 记录已显示的通知版本
      await chrome.storage.local.set({ lastUpdateNotification: currentVersion });
    } catch (error) {
      console.error('显示更新通知失败:', error);
    }
  }

  // 设置消息监听器
  setupMessageListener() {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      this.handleMessage(message, sender, sendResponse);
      return true; // 保持消息通道开放
    });
  }

  // 设置扩展图标点击监听器
  setupActionListener() {
    chrome.action.onClicked.addListener(async (tab) => {
      try {
        // 获取当前高亮状态
        const result = await chrome.storage.local.get(['highlightEnabled']);
        const currentState = result.highlightEnabled !== false; // 默认为true
        
        // 切换状态
        const newState = !currentState;
        await chrome.storage.local.set({ highlightEnabled: newState });
        
        // 通知所有标签页更新高亮状态
        const tabs = await chrome.tabs.query({});
        for (const tab of tabs) {
          try {
            await chrome.tabs.sendMessage(tab.id, {
              action: 'toggleHighlight',
              enabled: newState
            });
          } catch (error) {
            // 忽略无法发送消息的标签页（如chrome://页面）
          }
        }
        
        // 更新扩展图标状态
        await this.updateIconState(newState);
        
        console.log(`WzSLinker: 高亮功能已${newState ? '开启' : '关闭'}`);
        
        // 显示通知
        await this.showToggleNotification(newState);
        
      } catch (error) {
        console.error('WzSLinker: 切换高亮状态失败', error);
      }
    });
  }

  // 显示切换通知
  async showToggleNotification(enabled) {
    try {
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icon.png',
        title: 'WzSLinker 状态切换',
        message: `高亮功能已${enabled ? '开启' : '关闭'}`
      });
    } catch (error) {
      console.error('显示切换通知失败:', error);
    }
  }

  // 更新扩展图标状态
  async updateIconState(enabled) {
    try {
      if (enabled) {
        // 启用状态：使用正常图标，清除badge
        await chrome.action.setIcon({
          path: {
            "16": "icon.png",
            "32": "icon.png", 
            "48": "icon.png",
            "128": "icon.png"
          }
        });
        await chrome.action.setTitle({
          title: "WzSLinker - 高亮功能已启用"
        });
        await chrome.action.setBadgeText({ text: "" });
        await chrome.action.setBadgeBackgroundColor({ color: [0, 0, 0, 0] });
      } else {
        // 禁用状态：使用关闭图标，显示红色"OFF"badge
        await chrome.action.setIcon({
          path: {
            "16": "close16_2x.png",
            "32": "close16_2x.png", 
            "48": "close16_2x.png",
            "128": "close16_2x.png"
          }
        });
        await chrome.action.setTitle({
          title: "WzSLinker - 高亮功能已禁用"
        });
        await chrome.action.setBadgeText({ text: "OFF" });
        await chrome.action.setBadgeBackgroundColor({ color: [255, 51, 51, 255] }); // 红色
      }
      
      await chrome.storage.local.set({ 
        highlightEnabled: enabled,
        lastToggleTime: Date.now()
      });
    } catch (error) {
      console.error('WzSLinker: 更新图标状态失败', error);
    }
  }

  // 处理消息
  async handleMessage(message, sender, sendResponse) {
    try {
      switch (message.action) {
        case 'sendToLocalBridge':
          const result = await this.sendToLocalBridge(message.data);
          sendResponse({ success: true, data: result });
          break;
          
        case 'getStockData':
          const stockData = await this.getStockData();
          sendResponse({ success: true, data: stockData });
          break;
          
        case 'updateLinkHistory':
          await this.updateLinkHistory(message.historyItem);
          sendResponse({ success: true });
          break;
          
        case 'getLinkHistory':
          const history = await this.getLinkHistory();
          sendResponse({ success: true, data: history });
          break;
          
        case 'updateIconState':
          await this.updateIconState(message.enabled);
          sendResponse({ success: true });
          break;
          
        case 'updateHighlightCount':
          // 转发高亮计数到popup（如果popup打开的话）
          try {
            await chrome.storage.local.set({ currentHighlightCount: message.count });
          } catch (error) {
            console.error('保存高亮计数失败:', error);
          }
          sendResponse({ success: true });
          break;
          
        case 'diagnoseConnection':
          const diagnosticResults = await this.diagnoseConnection();
          sendResponse({ success: true, data: diagnosticResults });
          break;
          
        default:
          sendResponse({ success: false, error: 'Unknown action' });
      }
    } catch (error) {
      console.error('WzSLinker Background: 处理消息失败', error);
      sendResponse({ success: false, error: error.message });
    }
  }

  // 发送到Wzslinker本地服务（带重试机制）
  async sendToLocalBridge(data, retryCount = 0) {
    try {
      // 获取用户配置的服务地址、端口和超时设置
      const settings = await chrome.storage.local.get([
        'localServiceHost', 
        'localServicePort',
        'requestTimeout',
        'retryEnabled',
        'maxRetries'
      ]);
      
      const host = settings.localServiceHost || '127.0.0.1';
      const port = settings.localServicePort || '3000';
      const timeout = settings.requestTimeout || 5000; // 默认5秒
      const retryEnabled = settings.retryEnabled !== false; // 默认启用
      const maxRetries = settings.maxRetries || 2; // 默认最多重试2次
      const serviceUrl = `http://${host}:${port}/tdx`;
      
      console.log(`发送数据到Wzslinker本地服务... (尝试 ${retryCount + 1}/${maxRetries + 1})`, { url: serviceUrl, data });
      
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), timeout);
      
      try {
        const response = await fetch(serviceUrl, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(data),
          signal: controller.signal
        });

        clearTimeout(timeoutId);

        if (!response.ok) {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }

        const result = await response.json();
        console.log("联动广播请求成功发送。", result);
        
        // 成功后更新服务状态
        await chrome.storage.local.set({ 
          serviceStatus: 'online',
          lastServiceStatus: 'online'
        });
        
        return result;
      } catch (fetchError) {
        clearTimeout(timeoutId);
        
        // 如果启用重试且还有重试次数，则重试
        if (retryEnabled && retryCount < maxRetries) {
          console.log(`请求失败，${1000}ms后重试...`);
          await new Promise(resolve => setTimeout(resolve, 1000));
          return this.sendToLocalBridge(data, retryCount + 1);
        }
        
        throw fetchError;
      }
    } catch (error) {
      console.error("无法连接到Wzslinker本地服务。", error);
      
      // 分类错误并发送详细信息
      const errorDetails = this.categorizeError(error);
      await this.showDetailedError(errorDetails);
      
      // 向所有标签页发送错误消息，显示页面内通知
      try {
        const tabs = await chrome.tabs.query({});
        for (const tab of tabs) {
          try {
            await chrome.tabs.sendMessage(tab.id, {
              action: 'showServiceError',
              message: `无法连接到Wzslinker本地服务。${errorDetails.message}`,
              suggestion: errorDetails.suggestion
            });
          } catch (e) {
            // 忽略无法发送消息的标签页（如chrome://页面）
          }
        }
        console.log('页面内错误通知已发送到所有标签页');
      } catch (e) {
        console.error('无法发送错误消息到标签页:', e);
      }
      
      throw new Error(`联动失败：${errorDetails.message}`);
    }
  }

  // 获取股票数据
  async getStockData() {
    try {
      const result = await chrome.storage.local.get(['stockData']);
      return result.stockData || {};
    } catch (error) {
      console.error('WzSLinker Background: 获取股票数据失败', error);
      return {};
    }
  }

  // 更新联动历史
  async updateLinkHistory(historyItem) {
    try {
      const result = await chrome.storage.local.get(['linkHistory']);
      let history = result.linkHistory || [];
      
      // 添加新记录
      history.unshift({
        id: Date.now().toString(),
        ...historyItem,
        time: new Date().toLocaleString()
      });
      
      // 保留最近30条记录
      history = history.slice(0, 30);
      
      await chrome.storage.local.set({ linkHistory: history });
    } catch (error) {
      console.error('WzSLinker Background: 更新联动历史失败', error);
    }
  }

  // 获取联动历史
  async getLinkHistory() {
    try {
      const result = await chrome.storage.local.get(['linkHistory']);
      return result.linkHistory || [];
    } catch (error) {
      console.error('WzSLinker Background: 获取联动历史失败', error);
      return [];
    }
  }


  // 初始化默认数据
  async initializeDefaultData() {
    try {
      // 检查是否已有数据
      const result = await chrome.storage.local.get(['stockData', 'settings']);
      
      // 如果没有股票数据，尝试加载stock-data.json
      if (!result.stockData || Object.keys(result.stockData).length === 0) {
        let stockData = null;
        
        try {
          // 尝试从扩展目录加载stock-data.json
          stockData = await this.loadStockDataFromFile();
          console.log('WzSLinker: 成功从stock-data.json加载股票数据');
        } catch (error) {
          console.log('WzSLinker: 无法加载stock-data.json，使用默认数据', error);
          // 如果加载失败，使用默认数据
          stockData = await this.getDefaultStockData();
        }
        
        await chrome.storage.local.set({ stockData: stockData });
      }
      
      // 如果没有设置，初始化默认设置
      if (!result.settings) {
        const defaultSettings = {
          highlightStyle: 'none',
          highlightColor: '#fff3cd',
          textColor: '#ff0000',
          highlightEnabled: true,
          localServiceHost: '127.0.0.1',
          localServicePort: '3000'
        };
        await chrome.storage.local.set(defaultSettings);
      }
      
      // 初始化图标状态
      const highlightEnabled = result.highlightEnabled !== false; // 默认为true
      await this.updateIconState(highlightEnabled);
      
      console.log('WzSLinker: 默认数据初始化完成');
    } catch (error) {
      console.error('WzSLinker Background: 初始化默认数据失败', error);
    }
  }


  // 从扩展目录加载stock-data.json文件
  async loadStockDataFromFile() {
    return new Promise((resolve, reject) => {
      // 使用fetch加载扩展目录中的stock-data.json
      fetch(chrome.runtime.getURL('stock-data.json'))
        .then(response => {
          if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
          }
          return response.json();
        })
        .then(data => {
          if (typeof data === 'object' && data !== null) {
            resolve(data);
          } else {
            throw new Error('Invalid JSON format');
          }
        })
        .catch(error => {
          reject(error);
        });
    });
  }

  // 获取默认股票数据
  async getDefaultStockData() {
    // 这里返回一些常见的股票数据作为默认值
    // 实际使用时，用户可以通过导入JSON文件来更新
    return {
      "贵州茅台": "600519",
      "中国平安": "601318",
      "招商银行": "600036",
      "五粮液": "000858",
      "宁德时代": "300750",
      "比亚迪": "002594",
      "隆基绿能": "601012",
      "中信证券": "600030",
      "万科A": "000002",
      "格力电器": "000651",
      "东方财富": "300059",
      "恒瑞医药": "600276",
      "京东方A": "000725",
      "中国中免": "601888",
      "海康威视": "002415",
      "美的集团": "000333",
      "伊利股份": "600887",
      "中国建筑": "601668",
      "中国石化": "600028",
      "中国石油": "601857",
      "测试股票1": "999001",
      "测试股票2": "999002",
      "测试股票3": "999003"
    };
  }

  // 处理存储变化
  handleStorageChange(changes, namespace) {
    if (namespace === 'local') {
      // 当股票数据或设置发生变化时，通知所有标签页
      if (changes.stockData || changes.highlightEnabled || changes.highlightStyle) {
        this.notifyAllTabs('dataChanged', changes);
      }
    }
  }

  // 通知所有标签页
  async notifyAllTabs(action, data) {
    try {
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: action,
            data: data
          });
        } catch (error) {
          // 忽略无法发送消息的标签页（如chrome://页面）
        }
      }
    } catch (error) {
      console.error('WzSLinker Background: 通知标签页失败', error);
    }
  }

  // ==================== 更新检查功能 ====================
  
  // 设置通知点击监听器
  setupNotificationListener() {
    chrome.notifications.onClicked.addListener(async (notificationId) => {
      if (notificationId === 'update-available') {
        // 点击通知打开下载页面
        await chrome.tabs.create({
          url: 'https://hrfocus.top/extension-download'
        });
      }
    });

    chrome.notifications.onButtonClicked.addListener(async (notificationId, buttonIndex) => {
      if (notificationId === 'update-available') {
        switch (buttonIndex) {
          case 0: // 立即下载
            await this.openDownloadPage();
            break;
          case 1: // 查看详情
            await this.showUpdateDetails();
            break;
          case 2: // 稍后提醒
            await this.scheduleUpdateReminder();
            break;
        }
      }
    });
  }

  // 设置定期检查更新
  setupUpdateChecker() {
    // 每天检查一次更新（24小时 = 1440分钟）
    chrome.alarms.create('checkForUpdates', { 
      delayInMinutes: 1, 
      periodInMinutes: 1440 
    });
    
    chrome.alarms.onAlarm.addListener((alarm) => {
      if (alarm.name === 'checkForUpdates') {
        this.checkForUpdates();
      }
    });
    
    // 扩展启动时也检查一次
    this.checkForUpdates();
  }

  // 检查更新
  async checkForUpdates() {
    try {
      const currentVersion = chrome.runtime.getManifest().version;
      
      // 调用服务器API检查更新
      const response = await fetch('https://hrfocus.top/api/extension/check-update.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          currentVersion: currentVersion,
          extensionId: chrome.runtime.id,
          extensionName: 'WzSLinker'
        })
      });
      
      if (response.ok) {
        const updateInfo = await response.json();
        if (updateInfo.hasUpdate) {
          await this.showUpdateAvailableNotification(updateInfo);
        }
      }
    } catch (error) {
      console.error('检查更新失败:', error);
    }
  }

  // 显示有更新可用的通知
  async showUpdateAvailableNotification(updateInfo) {
    const notificationId = 'update-available';
    
    await chrome.notifications.create(notificationId, {
      type: 'basic',
      iconUrl: 'icon.png',
      title: 'WzSLinker 有新版本可用！',
      message: `版本 ${updateInfo.latestVersion} 已发布。点击下载更新。`,
      buttons: [
        { title: '立即下载' },
        { title: '查看详情' },
        { title: '稍后提醒' }
      ],
      requireInteraction: true
    });
    
    // 存储更新信息
    await chrome.storage.local.set({
      pendingUpdate: {
        version: updateInfo.latestVersion,
        downloadUrl: updateInfo.downloadUrl,
        changelog: updateInfo.changelog,
        notificationTime: Date.now()
      }
    });
  }

  // 打开下载页面
  async openDownloadPage() {
    const result = await chrome.storage.local.get(['pendingUpdate']);
    const updateInfo = result.pendingUpdate;
    
    if (updateInfo && updateInfo.downloadUrl) {
      await chrome.tabs.create({
        url: updateInfo.downloadUrl
      });
    } else {
      await chrome.tabs.create({
        url: 'https://hrfocus.top/extension-download'
      });
    }
    
    // 关闭通知
    chrome.notifications.clear('update-available');
  }

  // 显示更新详情
  async showUpdateDetails() {
    const result = await chrome.storage.local.get(['pendingUpdate']);
    const updateInfo = result.pendingUpdate;
    
    if (updateInfo && updateInfo.changelog) {
      const changelogHtml = Array.isArray(updateInfo.changelog) 
        ? updateInfo.changelog.map(item => `<li>${item}</li>`).join('') 
        : `<li>${updateInfo.changelog}</li>`;
      
      // 创建新标签页显示更新内容
      await chrome.tabs.create({
        url: `data:text/html;charset=utf-8,<html>
          <head>
            <meta charset="UTF-8">
            <title>WzSLinker ${updateInfo.version} 更新内容</title>
            <style>
              body { 
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
                padding: 40px; 
                line-height: 1.8; 
                max-width: 800px;
                margin: 0 auto;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: #333;
              }
              .container {
                background: white;
                border-radius: 12px;
                padding: 30px;
                box-shadow: 0 10px 40px rgba(0,0,0,0.2);
              }
              h1 { 
                color: #667eea; 
                border-bottom: 3px solid #667eea;
                padding-bottom: 15px;
                margin-bottom: 25px;
              }
              .changelog { 
                background: #f8f9fa; 
                padding: 20px; 
                border-radius: 8px; 
                border-left: 4px solid #667eea;
              }
              .changelog h3 {
                margin-top: 0;
                color: #495057;
              }
              .changelog ul {
                margin: 15px 0;
                padding-left: 20px;
              }
              .changelog li {
                margin: 10px 0;
                color: #495057;
              }
              .btn-group {
                margin-top: 30px;
                display: flex;
                gap: 15px;
              }
              .btn {
                padding: 12px 24px; 
                border: none; 
                border-radius: 6px; 
                cursor: pointer;
                font-size: 16px;
                font-weight: 500;
                transition: all 0.3s;
              }
              .btn-primary {
                background: #667eea; 
                color: white;
              }
              .btn-primary:hover {
                background: #5568d3;
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
              }
              .btn-secondary {
                background: #6c757d;
                color: white;
              }
              .btn-secondary:hover {
                background: #5a6268;
                transform: translateY(-2px);
              }
            </style>
          </head>
          <body>
            <div class="container">
              <h1>🎉 WzSLinker ${updateInfo.version} 更新内容</h1>
              <div class="changelog">
                <h3>✨ 新特性和改进：</h3>
                <ul>${changelogHtml}</ul>
              </div>
              <div class="btn-group">
                <button class="btn btn-primary" onclick="window.open('https://hrfocus.top/extension-download', '_blank')">立即下载</button>
                <button class="btn btn-secondary" onclick="window.close()">关闭</button>
              </div>
            </div>
          </body>
        </html>`
      });
    }
  }

  // 安排延迟提醒
  async scheduleUpdateReminder() {
    // 设置3小时后再次提醒
    const reminderTime = Date.now() + (3 * 60 * 60 * 1000);
    await chrome.storage.local.set({ 
      updateReminderTime: reminderTime
    });
    
    // 创建一个定时器，3小时后提醒
    chrome.alarms.create('updateReminder', {
      when: reminderTime
    });
    
    chrome.alarms.onAlarm.addListener((alarm) => {
      if (alarm.name === 'updateReminder') {
        this.checkUpdateReminder();
      }
    });
    
    // 关闭当前通知
    chrome.notifications.clear('update-available');
    
    // 显示延迟提醒确认
    chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icon.png',
      title: 'WzSLinker',
      message: '好的，3小时后再次提醒您更新'
    });
  }

  // 检查延迟提醒
  async checkUpdateReminder() {
    const result = await chrome.storage.local.get(['updateReminderTime', 'pendingUpdate']);
    
    if (result.updateReminderTime && result.updateReminderTime <= Date.now() && result.pendingUpdate) {
      await this.showUpdateAvailableNotification({
        latestVersion: result.pendingUpdate.version,
        downloadUrl: result.pendingUpdate.downloadUrl,
        changelog: result.pendingUpdate.changelog
      });
      // 清除提醒时间
      await chrome.storage.local.remove(['updateReminderTime']);
    }
  }

  // ==================== Service Worker Keepalive ====================
  
  // 设置 Service Worker 保活机制
  setupKeepalive() {
    // 使用 chrome.alarms 每1分钟唤醒一次 service worker
    chrome.alarms.create('keepalive', { 
      periodInMinutes: 1 
    });
    
    chrome.alarms.onAlarm.addListener((alarm) => {
      if (alarm.name === 'keepalive') {
        // 执行一个轻量级操作来保持活跃
        console.log('WzSLinker: Service worker keepalive ping');
      }
    });
  }

  // ==================== 健康检查定时器 ====================
  
  // 设置健康检查
  setupHealthCheck() {
    // 每5分钟检查一次服务健康状态
    chrome.alarms.create('healthCheck', { 
      delayInMinutes: 1,
      periodInMinutes: 5 
    });
    
    chrome.alarms.onAlarm.addListener((alarm) => {
      if (alarm.name === 'healthCheck') {
        this.checkServiceHealth();
      }
    });
  }

  // 检查服务健康状态
  async checkServiceHealth() {
    try {
      const settings = await chrome.storage.local.get(['localServiceHost', 'localServicePort']);
      const host = settings.localServiceHost || '127.0.0.1';
      const port = settings.localServicePort || '3000';
      
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3000);
      
      const response = await fetch(`http://${host}:${port}/status`, {
        method: 'GET',
        signal: controller.signal
      });
      
      clearTimeout(timeoutId);
      
      if (response.ok) {
        await chrome.storage.local.set({ 
          serviceStatus: 'online',
          lastHealthCheck: Date.now()
        });
        console.log('WzSLinker: 服务健康检查通过');
      }
    } catch (error) {
      await chrome.storage.local.set({ 
        serviceStatus: 'offline',
        lastHealthCheck: Date.now()
      });
      console.warn('WzSLinker: 服务健康检查失败', error.message);
      
      // 如果服务离线，显示友好提示
      if (await this.shouldNotifyServiceOffline()) {
        await this.showServiceOfflineNotification();
      }
    }
  }

  // 判断是否需要显示离线通知
  async shouldNotifyServiceOffline() {
    const result = await chrome.storage.local.get(['lastOfflineNotification', 'lastServiceStatus']);
    
    // 只在状态从在线变为离线时通知，且24小时内只通知一次
    const lastNotification = result.lastOfflineNotification || 0;
    const wasOnline = result.lastServiceStatus === 'online';
    
    if (wasOnline && (Date.now() - lastNotification > 24 * 60 * 60 * 1000)) {
      await chrome.storage.local.set({ 
        lastOfflineNotification: Date.now(),
        lastServiceStatus: 'offline'
      });
      return true;
    }
    
    return false;
  }

  // 显示服务离线通知
  async showServiceOfflineNotification() {
    try {
      await chrome.notifications.create('service-offline', {
        type: 'basic',
        iconUrl: 'icon.png',
        title: 'WzSLinker 服务离线',
        message: '检测到应用端已断开，请确保应用端正在运行'
      });
    } catch (error) {
      console.error('显示离线通知失败:', error);
    }
  }

  // ==================== 网络诊断工具 ====================
  
  // 网络诊断
  async diagnoseConnection() {
    const settings = await chrome.storage.local.get(['localServiceHost', 'localServicePort']);
    const host = settings.localServiceHost || '127.0.0.1';
    const port = settings.localServicePort || '3000';
    
    const diagnosticResults = {
      timestamp: new Date().toISOString(),
      host: host,
      port: port,
      tests: []
    };
    
    // 测试1: 基本连接测试
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3000);
      
      const response = await fetch(`http://${host}:${port}/status`, {
        method: 'GET',
        signal: controller.signal
      });
      
      clearTimeout(timeoutId);
      
      diagnosticResults.tests.push({
        name: '基本连接',
        status: 'SUCCESS',
        message: `成功连接到 ${host}:${port}`
      });
    } catch (error) {
      diagnosticResults.tests.push({
        name: '基本连接',
        status: 'FAILED',
        message: `无法连接到 ${host}:${port}`,
        error: error.message
      });
    }
    
    // 测试2: 联动接口测试
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3000);
      
      const response = await fetch(`http://${host}:${port}/tdx`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ code: '000001', name: '诊断测试', url: 'diagnostic' }),
        signal: controller.signal
      });
      
      clearTimeout(timeoutId);
      
      diagnosticResults.tests.push({
        name: '联动接口',
        status: 'SUCCESS',
        message: '联动接口响应正常'
      });
    } catch (error) {
      diagnosticResults.tests.push({
        name: '联动接口',
        status: 'FAILED',
        message: '联动接口无响应',
        error: error.message
      });
    }
    
    // 测试3: 检查端口配置
    diagnosticResults.tests.push({
      name: '端口检查',
      status: 'INFO',
      message: `配置端口: ${port}`,
      suggestion: '如果连接失败，请使用命令检查端口是否被占用: netstat -ano | findstr :' + port
    });
    
    return diagnosticResults;
  }

  // 错误分类
  categorizeError(error) {
    if (error.name === 'AbortError') {
      return {
        type: 'TIMEOUT',
        message: '连接超时，可能是应用端响应慢或未启动',
        suggestion: '请检查应用端是否正在运行，或增加超时时间'
      };
    } else if (error.message.includes('Failed to fetch') || 
               error.message.includes('ERR_CONNECTION_REFUSED')) {
      return {
        type: 'CONNECTION_REFUSED',
        message: '连接被拒绝，应用端未运行',
        suggestion: '请启动 WzSLinker 应用端'
      };
    } else if (error.message.includes('ERR_BLOCKED_BY_CLIENT') ||
               error.message.includes('CORS')) {
      return {
        type: 'BLOCKED',
        message: '请求被浏览器阻止',
        suggestion: '请检查浏览器安全设置或防火墙'
      };
    } else {
      return {
        type: 'UNKNOWN',
        message: error.message || '未知错误',
        suggestion: '请查看控制台获取详细错误信息'
      };
    }
  }

  // 显示详细错误
  async showDetailedError(errorDetails) {
    try {
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'showDetailedError',
            errorType: errorDetails.type,
            message: errorDetails.message,
            suggestion: errorDetails.suggestion
          });
        } catch (e) {
          // 忽略无法发送消息的标签页
        }
      }
    } catch (e) {
      console.error('无法发送错误消息到标签页:', e);
    }
  }
}

// 初始化后台服务
const wzsLinkerBackground = new WzSLinkerBackground();

// 监听存储变化
chrome.storage.onChanged.addListener((changes, namespace) => {
  wzsLinkerBackground.handleStorageChange(changes, namespace);
});

