// WzSLinker 内容脚本 - 核心识别逻辑

/**
 * WzSLinker内容脚本类
 * 负责在网页中识别股票代码和名称，并提供高亮显示和联动功能
 */
class WzSLinkerContentScript {
  constructor() {
    this.stockData = {};
    this.settings = {
      highlightStyle: 'none',
      highlightColor: '#fff3cd',
      textColor: '#ff0000',
      highlightEnabled: true,
      showPriceEnabled: true  // 价格显示开关
    };
    this.processedNodes = new WeakSet();
    this.observer = null;
    this.intersectionObserver = null; // 新增：可见性观察器
    this.isProcessing = false;
    this.batchSize = 50;
    this.delay = 30; // 优化：从100ms降低到30ms，提升响应速度
    this.maxWaitTime = 300; // 新增：最大等待时间300ms，避免频繁更新时延迟累积
    this.lastProcessTime = 0; // 新增：记录上次处理时间
    this.highlightCount = 0;
    
    // 价格缓存系统
    this.priceCache = new Map();
    this.cacheTimeout = 30000; // 30秒缓存
    
    // 请求控制（优化：针对100+股票场景）
    this.requestQueue = new Map();
    this.requestInProgress = new Set();
    this.requestRateLimit = 50; // 50ms请求间隔（大幅降低，提升速度）
    this.maxConcurrentRequests = 20; // 最大并发数（大幅提升到20，支持100+股票）
    this.lastRequestTime = 0;
    
    // 批量请求配置（新增：支持批量查询）
    this.batchSize = 10; // 每批10个股票
    this.batchQueue = [];
    this.batchTimeout = null;
    
    // 定时更新
    this.priceUpdateInterval = null;
    
    // User-Agent 配置（优化：使用多个UA提高成功率和速度）
    this.userAgents = {
      // 移动端UA - 通常响应更快，数据更轻量
      mobile: 'Mozilla/5.0 (iPhone; CPU iPhone OS 16_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.0 Mobile/15E148 Safari/604.1',
      // 桌面Chrome - 兼容性最好
      chrome: 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
      // 桌面Edge - 某些国内网站优化
      edge: 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36 Edg/120.0.0.0',
      // Firefox - 备用
      firefox: 'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:121.0) Gecko/20100101 Firefox/121.0'
    };
    
    this.init();
  }

  async init() {
    await this.loadData();
    
    // 检查当前域名是否被允许运行
    if (!this.isDomainAllowed()) {
      console.log('WzSLinker: 当前域名在黑名单或不在白名单中，扩展功能已禁用');
      return;
    }
    
    this.setupMessageListener();
    this.setupIntersectionObserver();
    this.startObserving();
    this.lastProcessTime = Date.now(); // 初始化处理时间
    this.processPage();
    this.startPriceUpdates(); // 启动定时更新
  }

  /**
   * 加载股票数据和设置
   * 从Chrome存储中获取股票数据和用户设置
   */
  async loadData() {
    try {
      const result = await chrome.storage.local.get([
        'stockData',
        'highlightEnabled',
        'highlightStyle',
        'highlightColor',
        'textColor',
        'showPriceEnabled',
        'domainBlocklistEnabled',
        'domainBlocklist',
        'domainAllowlistEnabled',
        'domainAllowlist'
      ]);
      
      // 验证股票数据格式
      let stockData = result.stockData || {};
      
      // 数据验证：确保是对象且包含有效数据
      if (typeof stockData !== 'object' || stockData === null || Array.isArray(stockData)) {
        console.warn('WzSLinker: 股票数据格式错误，尝试恢复默认数据');
        stockData = await this.recoverStockData();
      }
      
      // 检查数据是否为空
      if (Object.keys(stockData).length === 0) {
        console.warn('WzSLinker: 股票数据为空，尝试从background加载');
        const response = await chrome.runtime.sendMessage({ action: 'getStockData' });
        if (response && response.success && response.data) {
          stockData = response.data;
        }
      }
      
      this.stockData = stockData;
      this.settings = {
        highlightStyle: result.highlightStyle || 'none',
        highlightColor: result.highlightColor || '#fff3cd',
        textColor: result.textColor || '#ff0000',
        highlightEnabled: result.highlightEnabled !== undefined ? result.highlightEnabled : true,
        showPriceEnabled: result.showPriceEnabled !== undefined ? result.showPriceEnabled : true,
        domainBlocklistEnabled: result.domainBlocklistEnabled || false,
        domainBlocklist: result.domainBlocklist || [],
        domainAllowlistEnabled: result.domainAllowlistEnabled || false,
        domainAllowlist: result.domainAllowlist || []
      };
      
      this.buildRegex();
      this.updateHighlightStyles();
      
      // 记录加载成功的日志
      console.log(`WzSLinker: 数据加载完成，包含 ${Object.keys(this.stockData).length} 条股票数据`);
      
    } catch (error) {
      console.error('WzSLinker: 加载数据失败', error);
      // 尝试恢复
      await this.recoverStockData();
    }
  }

  /**
   * 恢复股票数据
   * 当数据损坏或丢失时，尝试从background service获取默认数据
   */
  async recoverStockData() {
    try {
      console.log('WzSLinker: 开始恢复股票数据...');
      
      // 尝试从background service获取默认数据
      const response = await chrome.runtime.sendMessage({ action: 'getStockData' });
      if (response && response.success && response.data) {
        const stockData = response.data;
        
        // 验证恢复的数据
        if (typeof stockData === 'object' && stockData !== null && !Array.isArray(stockData)) {
          await chrome.storage.local.set({ stockData: stockData });
          console.log(`WzSLinker: 成功恢复 ${Object.keys(stockData).length} 条股票数据`);
          return stockData;
        }
      }
      
      console.warn('WzSLinker: 无法从background恢复数据，使用空数据集');
      return {};
    } catch (error) {
      console.error('WzSLinker: 无法恢复股票数据', error);
      return {};
    }
  }

  // 构建正则表达式
  buildRegex() {
    if (!this.stockData || Object.keys(this.stockData).length === 0) {
      this.stockRegex = null;
      this.stockCodeRegex = null;
      this.stockNameRegex = null;
      return;
    }

    const stockNames = Object.keys(this.stockData);
    const stockCodes = Object.values(this.stockData);
    
    // 股票代码正则：匹配6位数字代码
    this.stockCodeRegex = /\b(00[0-9]{4}|30[0-9]{4}|60[0-9]{4}|68[0-9]{4}|92[0-9]{4})\b/g;
    
    // 股票名称正则：转义特殊字符并构建模式（支持字符间的空格）
    // 先拆分字符，再分别转义，最后用可选空格连接
    const escapedStockNames = stockNames.map(name => {
      // 使用 Array.from 正确处理 Unicode 字符（在转义之前拆分）
      const chars = Array.from(name);
      if (chars.length === 1) {
        // 单字符名称，直接转义返回
        return this.escapeRegex(chars[0]);
      }
      // 多字符名称，分别转义每个字符，然后用可选空格连接
      const escapedChars = chars.map(char => this.escapeRegex(char));
      return escapedChars.join('\\s{0,3}');
    });
    
    this.stockNameRegex = new RegExp(`(${escapedStockNames.join('|')})`, 'g');
    
    // 组合正则：同时匹配股票代码和名称
    const pattern = `${this.stockCodeRegex.source}|${this.stockNameRegex.source}`;
    this.stockRegex = new RegExp(pattern, 'g');
    
    console.log(`WzSLinker: 正则表达式已构建，支持 ${stockNames.length} 个股票名称（含空格匹配）`);
  }

  // 转义正则表达式特殊字符
  escapeRegex(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
  }

  // 设置IntersectionObserver（性能优化）
  setupIntersectionObserver() {
    // 只处理可见区域的元素
    this.intersectionObserver = new IntersectionObserver((entries) => {
      for (const entry of entries) {
        if (entry.isIntersecting && this.settings.highlightEnabled) {
          // 元素进入视口，处理它的文本节点
          this.processElement(entry.target);
          // 处理完后取消观察
          this.intersectionObserver.unobserve(entry.target);
        }
      }
    }, {
      rootMargin: '100px', // 提前100px开始处理
      threshold: 0.01 // 元素至少1%可见时触发
    });
  }

  // 处理单个元素（用于IntersectionObserver）
  processElement(element) {
    if (!this.settings.highlightEnabled || !this.stockRegex) {
      return;
    }

    const walker = document.createTreeWalker(
      element,
      NodeFilter.SHOW_TEXT,
      {
        acceptNode: (node) => {
          if (this.processedNodes.has(node) || 
              node.parentElement?.tagName === 'SCRIPT' ||
              node.parentElement?.tagName === 'STYLE' ||
              node.parentElement?.tagName === 'STOCK-HIGHLIGHT') {
            return NodeFilter.FILTER_REJECT;
          }
          return NodeFilter.FILTER_ACCEPT;
        }
      }
    );

    const textNodes = [];
    let node;
    while (node = walker.nextNode()) {
      textNodes.push(node);
    }

    // 处理文本节点
    for (const textNode of textNodes) {
      this.processTextNode(textNode);
    }
    
    this.updateHighlightCount();
  }

  // 设置消息监听器
  setupMessageListener() {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      console.log('WzSLinker: 收到消息', message);
      
      switch (message.action) {
        case 'ping':
          console.log('WzSLinker: 响应ping消息');
          sendResponse({ status: 'ready', timestamp: Date.now() });
          return true; // 保持消息通道开放
          
        case 'toggleHighlight':
          this.settings.highlightEnabled = message.enabled;
          this.toggleHighlight(message.enabled);
          sendResponse({ success: true });
          break;
          
        case 'changeHighlightStyle':
          this.settings.highlightStyle = message.style;
          this.updateHighlightStyles();
          sendResponse({ success: true });
          break;
          
        case 'reloadStockData':
          this.stockData = message.data || {};
          this.buildRegex();
          this.processPage();
          sendResponse({ success: true });
          break;
          
        case 'updateSettings':
          console.log('WzSLinker: 收到设置更新', message.settings);
          const oldHighlightEnabled = this.settings.highlightEnabled;
          const oldShowPriceEnabled = this.settings.showPriceEnabled;
          Object.assign(this.settings, message.settings);
          console.log('WzSLinker: 更新后的设置', this.settings);
          
          // 如果高亮开关被关闭，移除所有高亮
          if (message.settings.highlightEnabled === false && oldHighlightEnabled !== false) {
            console.log('WzSLinker: 高亮开关被关闭，移除所有高亮');
            this.removeAllHighlights();
          } else {
            // 如果只是颜色或样式变化，只更新现有高亮
            if (message.settings.highlightColor || message.settings.textColor || message.settings.highlightStyle) {
              console.log('WzSLinker: 颜色或样式变化，更新现有高亮');
              this.updateHighlightStyles();
            }
            // 如果高亮开关被打开，重新处理页面
            if (message.settings.highlightEnabled === true && oldHighlightEnabled === false) {
              console.log('WzSLinker: 高亮开关被打开，重新处理页面');
              this.processPage();
            }
          }
          
          // 处理涨跌幅显示开关变化
          if (message.settings.showPriceEnabled !== undefined) {
            if (message.settings.showPriceEnabled === false && oldShowPriceEnabled !== false) {
              console.log('WzSLinker: 涨跌幅显示被关闭，移除所有价格指示器');
              this.removeAllPriceIndicators();
            } else if (message.settings.showPriceEnabled === true && oldShowPriceEnabled === false) {
              console.log('WzSLinker: 涨跌幅显示被打开，重新添加价格指示器');
              this.restorePriceIndicators();
            }
          }
          
          sendResponse({ success: true });
          break;
          
        case 'resetSettings':
          this.loadData();
          this.processPage();
          sendResponse({ success: true });
          break;
          
        case 'updateDomainSettings':
          console.log('WzSLinker: 收到域名设置更新', message.settings);
          Object.assign(this.settings, message.settings);
          
          // 检查当前域名是否被允许运行
          if (!this.isDomainAllowed()) {
            console.log('WzSLinker: 域名设置更新后，当前域名不再被允许，禁用功能');
            // 移除所有高亮和价格指示器
            this.removeAllHighlights();
            this.removeAllPriceIndicators();
            // 停止观察
            if (this.observer) {
              this.observer.disconnect();
            }
          } else {
            console.log('WzSLinker: 域名设置更新后，当前域名被允许，功能正常');
            // 如果之前被禁用，现在允许了，重新初始化
            if (!this.observer) {
              this.setupIntersectionObserver();
              this.startObserving();
              this.processPage();
              this.startPriceUpdates();
            }
          }
          
          sendResponse({ success: true });
          break;
          
        case 'linkStock':
          console.log('WzSLinker: 收到联动请求', { code: message.code, name: message.name });
          this.handleDirectLink(message.code, message.name)
            .then(result => {
              console.log('WzSLinker: 联动完成，发送响应', result);
              sendResponse({ success: true, result: result });
            })
            .catch(error => {
              console.error('WzSLinker: 联动失败，发送错误响应', error);
              sendResponse({ success: false, error: error.message });
            });
          return true; // 保持消息通道开放以等待异步响应
          
        case 'processPage':
          this.processPage();
          sendResponse({ success: true });
          break;
          
        case 'showServiceError':
          // 显示服务错误通知
          this.showServiceErrorNotification(message.message, message.suggestion);
          sendResponse({ success: true });
          break;
          
        case 'showDetailedError':
          // 显示详细错误信息
          this.showDetailedErrorNotification(message.errorType, message.message, message.suggestion);
          sendResponse({ success: true });
          break;
          
        case 'testNotification':
          // 测试通知功能
          this.testNotification();
          sendResponse({ success: true });
          break;
          
        default:
          console.warn('WzSLinker: 未知消息类型', message.action);
          sendResponse({ success: false, error: 'Unknown action' });
      }
    });
  }

  // 开始观察DOM变化
  startObserving() {
    if (this.observer) {
      this.observer.disconnect();
    }

    this.observer = new MutationObserver((mutations) => {
      if (this.isProcessing) return;
      
      let shouldProcess = false;
      for (const mutation of mutations) {
        if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
          shouldProcess = true;
          break;
        }
      }
      
      if (shouldProcess) {
        const now = Date.now();
        const timeSinceLastProcess = now - this.lastProcessTime;
        
        // 如果距离上次处理超过最大等待时间，立即处理
        if (timeSinceLastProcess >= this.maxWaitTime) {
          clearTimeout(this.processTimeout);
          this.lastProcessTime = now;
          this.processPage();
        } else {
          // 防抖处理：使用较短的延迟
          clearTimeout(this.processTimeout);
          this.processTimeout = setTimeout(() => {
            this.lastProcessTime = Date.now();
            this.processPage();
          }, this.delay);
        }
      }
    });

    this.observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }

  // 处理页面（使用IntersectionObserver优化）
  async processPage() {
    if (!this.settings.highlightEnabled || !this.stockRegex) {
      return;
    }

    this.isProcessing = true;
    
    try {
      // 首先恢复现有元素的价格指示器
      this.restorePriceIndicators();
      
      // 使用TreeWalker高效遍历DOM
      const walker = document.createTreeWalker(
        document.body,
        NodeFilter.SHOW_TEXT,
        {
          acceptNode: (node) => {
            // 跳过已处理的节点和脚本/样式节点
            if (this.processedNodes.has(node) || 
                node.parentElement?.tagName === 'SCRIPT' ||
                node.parentElement?.tagName === 'STYLE' ||
                node.parentElement?.tagName === 'STOCK-HIGHLIGHT') {
              return NodeFilter.FILTER_REJECT;
            }
            
            return NodeFilter.FILTER_ACCEPT;
          }
        }
      );

      const visibleTextNodes = [];
      const invisibleElements = new Set();
      let node;
      
      while (node = walker.nextNode()) {
        const parent = node.parentElement;
        
        if (!parent) continue;
        
        // 检查元素是否在视口内或附近
        const rect = parent.getBoundingClientRect();
        const isNearViewport = rect.top < window.innerHeight + 200 && rect.bottom > -200;
        
        if (isNearViewport) {
          // 可见或接近可见的元素，立即处理
          visibleTextNodes.push(node);
        } else {
          // 不可见的元素，添加到IntersectionObserver观察列表
          if (this.intersectionObserver && !invisibleElements.has(parent)) {
            invisibleElements.add(parent);
            this.intersectionObserver.observe(parent);
          }
        }
      }

      // 优先批量处理可见区域的文本节点
      await this.processTextNodesBatch(visibleTextNodes);
      
      // 更新高亮计数
      this.updateHighlightCount();
      
      console.log(`WzSLinker: 已处理 ${visibleTextNodes.length} 个可见文本节点，${invisibleElements.size} 个元素添加到延迟加载队列`);
      
    } catch (error) {
      console.error('WzSLinker: 处理页面失败', error);
    } finally {
      this.isProcessing = false;
    }
  }

  // 批量处理文本节点
  async processTextNodesBatch(textNodes) {
    for (let i = 0; i < textNodes.length; i += this.batchSize) {
      const batch = textNodes.slice(i, i + this.batchSize);
      
      for (const textNode of batch) {
        this.processTextNode(textNode);
      }
      
      // 让出控制权，避免阻塞UI
      if (i + this.batchSize < textNodes.length) {
        await new Promise(resolve => setTimeout(resolve, 0));
      }
    }
  }

  // 处理单个文本节点
  processTextNode(textNode) {
    const text = textNode.textContent;
    if (!text || text.length < 2) return;

    const matches = [...text.matchAll(this.stockRegex)];
    if (matches.length === 0) return;

    // 标记为已处理
    this.processedNodes.add(textNode);

    // 创建高亮元素
    this.createHighlights(textNode, matches);
  }

  // 创建高亮元素
  createHighlights(textNode, matches) {
    const parent = textNode.parentNode;
    if (!parent) return;

    let lastIndex = 0;
    const fragment = document.createDocumentFragment();

    for (const match of matches) {
      const matchText = match[0];
      const matchIndex = match.index;
      
      // 添加匹配前的文本
      if (matchIndex > lastIndex) {
        const beforeText = textNode.textContent.substring(lastIndex, matchIndex);
        fragment.appendChild(document.createTextNode(beforeText));
      }

      // 创建高亮元素
      const highlightElement = this.createHighlightElement(matchText);
      fragment.appendChild(highlightElement);

      lastIndex = matchIndex + matchText.length;
    }

    // 添加剩余的文本
    if (lastIndex < textNode.textContent.length) {
      const remainingText = textNode.textContent.substring(lastIndex);
      fragment.appendChild(document.createTextNode(remainingText));
    }

    // 替换原文本节点
    parent.replaceChild(fragment, textNode);
  }

  // 创建高亮元素
  createHighlightElement(text) {
    const element = document.createElement('stock-highlight');
    element.setAttribute('data-stock-text', text);
    
    // 获取股票信息
    const stockCode = this.getStockCode(text);
    const stockName = this.getStockName(text);
    
    // 设置显示内容：保持原始文本显示
    element.textContent = text;
    
    // 添加股票信息到数据属性
    if (stockName && stockCode) {
      element.setAttribute('data-stock-name', stockName);
      element.setAttribute('data-stock-code', stockCode);
    }
    
    // 应用样式
    this.applyHighlightStyle(element);
    
    // 添加点击事件
    element.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      this.handleStockClick(text);
    });

    // 增加高亮计数
    this.highlightCount++;
    this.updateHighlightCount();
    
    // 添加价格指示器
    if (this.settings.showPriceEnabled && stockCode) {
      setTimeout(() => {
        this.addPriceIndicator(element, stockCode);
      }, 100);
    }

    return element;
  }

  // 处理股票点击
  async handleStockClick(text) {
    const stockCode = this.getStockCode(text);
    const stockName = this.getStockName(text);
    
    if (!stockCode || !stockName) {
      console.warn('WzSLinker: 无法获取股票信息', { text, stockCode, stockName });
      return;
    }

    try {
      // 发送联动请求
      await this.linkStock(stockCode, stockName);
      
      // 记录联动历史
      await this.recordLinkHistory(stockCode, stockName);
      
      // 显示成功提示
      this.showNotification(`已联动: ${stockName} (${stockCode})`);
      
    } catch (error) {
      console.error('WzSLinker: 联动失败', error);
      
      // 检查是否是本地服务连接失败
      if (error.message.includes('无法连接到Wzslinker本地服务') || 
          error.message.includes('Failed to fetch') ||
          error.message.includes('ERR_CONNECTION_REFUSED')) {
        console.error('WzSLinker: Wzslinker本地服务连接失败，请确保Wzslinker本地服务正在运行');
        // 显示服务错误通知（带官网链接）
        this.showServiceErrorNotification('无法连接到Wzslinker本地服务。请检查本地服务是否正在运行。');
      } else {
        // 其他错误显示普通通知
        this.showNotification('联动失败，请检查本地服务', 'error');
      }
    }
  }

  // 处理直接联动请求（来自popup）
  async handleDirectLink(code, name) {
    console.log('WzSLinker: 开始处理直接联动', { code, name });
    
    if (!code || !name) {
      console.error('WzSLinker: 联动参数不完整', { code, name });
      throw new Error('联动参数不完整');
    }

    try {
      // 发送联动请求
      console.log('WzSLinker: 准备发送联动请求', { code, name });
      const linkResult = await this.linkStock(code, name);
      
      // 记录联动历史
      await this.recordLinkHistory(code, name);
      
      // 显示成功提示
      this.showNotification(`已联动: ${name} (${code})`);
      
      console.log('WzSLinker: 直接联动完成', { code, name, result: linkResult });
      
      return { success: true, code, name, result: linkResult };
      
    } catch (error) {
      console.error('WzSLinker: 直接联动失败', error);
      
      // 检查是否是本地服务连接失败
      if (error.message.includes('无法连接到Wzslinker本地服务') || 
          error.message.includes('Failed to fetch') ||
          error.message.includes('ERR_CONNECTION_REFUSED')) {
        console.error('WzSLinker: Wzslinker本地服务连接失败，请确保Wzslinker本地服务正在运行');
        // 显示服务错误通知（带官网链接）
        this.showServiceErrorNotification('无法连接到Wzslinker本地服务。请检查本地服务是否正在运行。');
      } else {
        // 其他错误显示普通通知
        this.showNotification('联动失败，请检查本地服务', 'error');
      }
      
      throw error; // 重新抛出错误以便popup处理
    }
  }

  // 获取股票代码
  getStockCode(text) {
    // 如果是6位数字代码，直接返回
    if (/^\d{6}$/.test(text)) {
      return text;
    }
    
    // 移除文本中的所有空白符，然后从股票名称获取代码
    const normalizedText = text.replace(/\s+/g, '');
    
    // 先尝试直接匹配
    if (this.stockData[normalizedText]) {
      return this.stockData[normalizedText];
    }
    
    // 再尝试原始文本匹配（兼容旧逻辑）
    return this.stockData[text];
  }

  // 获取股票名称
  getStockName(text) {
    // 移除文本中的所有空白符进行规范化
    const normalizedText = text.replace(/\s+/g, '');
    
    // 如果规范化后的文本是股票名称，返回规范化的名称
    if (this.stockData[normalizedText]) {
      return normalizedText;
    }
    
    // 尝试原始文本（兼容旧逻辑）
    if (this.stockData[text]) {
      return text;
    }
    
    // 从代码获取名称
    for (const [name, code] of Object.entries(this.stockData)) {
      if (code === text) {
        return name;
      }
    }
    
    return null;
  }

  // 判断是否为股票代码
  isStockCode(text) {
    return /^\d{6}$/.test(text);
  }

  // 判断是否为股票名称
  isStockName(text) {
    return this.stockData.hasOwnProperty(text);
  }

  // 联动股票 - 通过background.js发送到Wzslinker本地服务
  async linkStock(code, name) {
    try {
      console.log(`发送代码 ${code} 到Wzslinker本地服务进行广播...`);
      
      // 通过消息传递机制调用background.js的sendToLocalBridge函数
      const response = await chrome.runtime.sendMessage({
        action: 'sendToLocalBridge',
        data: {
          code: code,
          name: name,
          url: window.location.href
        }
      });

      if (!response.success) {
        throw new Error(response.error || '联动失败');
      }

      console.log("联动广播请求成功发送。", response.data);
      return response.data;
    } catch (error) {
      console.error("无法连接到Wzslinker本地服务。", error);
      throw new Error(`联动失败：无法连接到Wzslinker本地服务。${error.message}`);
    }
  }

  // 记录联动历史
  async recordLinkHistory(code, name) {
    try {
      const result = await chrome.storage.local.get(['linkHistory']);
      let history = result.linkHistory || [];
      
      const historyItem = {
        id: Date.now().toString(),
        code: code,
        name: name,
        time: new Date().toLocaleString(),
        url: window.location.href
      };
      
      history.unshift(historyItem);
      history = history.slice(0, 30); // 保留最近30条
      
      await chrome.storage.local.set({ linkHistory: history });
    } catch (error) {
      console.error('WzSLinker: 记录历史失败', error);
    }
  }

  /**
   * 切换高亮显示
   * @param {boolean} enabled - 是否启用高亮
   */
  toggleHighlight(enabled) {
    this.settings.highlightEnabled = enabled;
    
    if (enabled) {
      this.processPage();
    } else {
      this.removeAllHighlights();
    }
  }


  // 更新所有高亮样式（包括颜色）
  updateHighlightStyles() {
    const highlights = document.querySelectorAll('stock-highlight');
    highlights.forEach(highlight => {
      this.applyHighlightStyle(highlight);
    });
  }

  // 应用高亮样式到单个元素
  applyHighlightStyle(element) {
    const style = this.settings.highlightStyle;
    const highlightColor = this.settings.highlightColor;
    const textColor = this.settings.textColor;
    
    // 设置基础样式
    element.className = `stock-highlight ${style}`;
    
    // 根据样式类型应用不同的CSS属性 - 使用 setProperty 和 important 优先级覆盖CSS
    switch (style) {
      case 'normal':
        element.style.setProperty('background-color', highlightColor, 'important');
        element.style.setProperty('color', textColor, 'important');
        element.style.setProperty('box-shadow', 'none', 'important');
        element.style.setProperty('border', 'none', 'important');
        element.style.setProperty('text-decoration', 'none', 'important');
        element.style.setProperty('border-bottom', 'none', 'important');
        element.style.setProperty('font-weight', 'bold', 'important');
        break;
        
      case 'glow':
        const rgb = this.hexToRgb(highlightColor);
        element.style.setProperty('background-color', `rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.3)`, 'important');
        element.style.setProperty('color', textColor, 'important');
        element.style.setProperty('box-shadow', `0 0 10px rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.8)`, 'important');
        element.style.setProperty('border', `1px solid rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.5)`, 'important');
        element.style.setProperty('text-decoration', 'none', 'important');
        element.style.setProperty('border-bottom', 'none', 'important');
        element.style.setProperty('font-weight', 'bold', 'important');
        break;
        
      case 'none':
        element.style.setProperty('background-color', 'transparent', 'important');
        element.style.setProperty('color', textColor, 'important');
        element.style.setProperty('box-shadow', 'none', 'important');
        element.style.setProperty('border', 'none', 'important');
        element.style.setProperty('text-decoration', `underline dotted ${textColor}`, 'important');
        element.style.setProperty('border-bottom', `1px dotted ${textColor}`, 'important');
        element.style.setProperty('font-weight', 'normal', 'important');
        break;
    }
  }

  // 十六进制颜色转RGB（使用公共工具函数）
  hexToRgb(hex) {
    return window.WzSLinkerUtils ? window.WzSLinkerUtils.hexToRgb(hex) : this._hexToRgb(hex);
  }

  // 备用十六进制颜色转RGB函数
  _hexToRgb(hex) {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? {
      r: parseInt(result[1], 16),
      g: parseInt(result[2], 16),
      b: parseInt(result[3], 16)
    } : { r: 255, g: 243, b: 205 }; // 默认颜色
  }

  // 移除所有高亮
  removeAllHighlights() {
    const highlights = document.querySelectorAll('stock-highlight');
    highlights.forEach(highlight => {
      const parent = highlight.parentNode;
      if (parent) {
        parent.replaceChild(document.createTextNode(highlight.textContent), highlight);
        parent.normalize();
      }
    });
    
    // 断开IntersectionObserver
    if (this.intersectionObserver) {
      this.intersectionObserver.disconnect();
      // 重新创建observer以便下次使用
      this.setupIntersectionObserver();
    }
    
    this.processedNodes = new WeakSet();
    this.highlightCount = 0;
    this.updateHighlightCount();
  }

  // 更新高亮计数
  updateHighlightCount() {
    // 实际计算页面中的高亮元素数量
    const actualCount = document.querySelectorAll('stock-highlight').length;
    this.highlightCount = actualCount;
    
    // 通知popup更新计数显示
    chrome.runtime.sendMessage({
      action: 'updateHighlightCount',
      count: this.highlightCount
    }).catch(() => {
      // 忽略发送失败的情况
    });
  }

  // 显示通知
  showNotification(message, type = 'success') {
    // 移除现有通知
    const existing = document.querySelector('.wzslinker-notification');
    if (existing) {
      existing.remove();
    }

    // 创建新通知
    const notification = document.createElement('div');
    notification.className = `wzslinker-notification ${type}`;
    notification.textContent = message;
    
    // 普通样式配置
    const normalConfig = {
      success: {
        background: '#fff3cd',
        color: '#ff0000',
        border: '1px solid #ffe082'
      },
      error: {
        background: '#f8d7da',
        color: '#721c24',
        border: '1px solid #f5c6cb'
      }
    };
    
    const config = normalConfig[type] || normalConfig.success;
    
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: ${config.background};
      color: ${config.color};
      padding: 12px 20px;
      border-radius: 6px;
      font-size: 14px;
      font-weight: bold;
      z-index: 1000000;
      border: ${config.border};
      box-shadow: 0 4px 12px rgba(0,0,0,0.3);
      animation: slideIn 0.3s ease-out;
    `;

    // 添加动画样式
    if (!document.querySelector('#wzslinker-notification-styles')) {
      const style = document.createElement('style');
      style.id = 'wzslinker-notification-styles';
      style.textContent = `
        @keyframes slideIn {
          from { transform: translateX(100%); opacity: 0; }
          to { transform: translateX(0); opacity: 1; }
        }
        @keyframes slideOut {
          from { transform: translateX(0); opacity: 1; }
          to { transform: translateX(100%); opacity: 0; }
        }
      `;
      document.head.appendChild(style);
    }

    document.body.appendChild(notification);

    // 3秒后自动移除
    setTimeout(() => {
      if (notification.parentNode) {
        notification.style.animation = 'slideOut 0.3s ease-in';
        setTimeout(() => {
          if (notification.parentNode) {
            notification.remove();
          }
        }, 300);
      }
    }, 3000);
  }

  // 测试通知功能
  testNotification() {
    console.log('WzSLinker: 测试通知功能');
    this.showServiceErrorNotification('这是一个测试通知，用于验证页面内通知功能是否正常工作。');
  }

  // 显示详细错误通知（带诊断按钮）
  showDetailedErrorNotification(errorType, message, suggestion) {
    // 移除现有通知
    const existing = document.querySelector('.wzslinker-service-error');
    if (existing) {
      existing.remove();
    }

    // 创建错误通知容器
    const notification = document.createElement('div');
    notification.className = 'wzslinker-service-error';
    
    // 根据错误类型选择图标和颜色
    const errorConfig = {
      'TIMEOUT': { icon: '⏱️', color: '#ff9800' },
      'CONNECTION_REFUSED': { icon: '🚫', color: '#f44336' },
      'BLOCKED': { icon: '🛡️', color: '#9c27b0' },
      'UNKNOWN': { icon: '❓', color: '#607d8b' }
    };
    
    const config = errorConfig[errorType] || errorConfig['UNKNOWN'];
    
    // 创建通知内容
    notification.innerHTML = `
      <div class="error-header" style="background: ${config.color}">
        <span class="error-icon">${config.icon}</span>
        <span class="error-title">WzSLinker 联动失败</span>
        <button class="error-close">×</button>
      </div>
      <div class="error-content">
        <p><strong>错误：</strong>${message}</p>
        ${suggestion ? `<p class="error-suggestion"><strong>建议：</strong>${suggestion}</p>` : ''}
        <div class="error-buttons">
          <button class="error-button error-button-primary">访问官网</button>
          <button class="error-button error-button-secondary" id="diagnosticBtn">运行诊断</button>
        </div>
      </div>
    `;
    
    // 设置样式
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: #fff;
      color: #333;
      border: 2px solid ${config.color};
      border-radius: 8px;
      padding: 0;
      font-size: 14px;
      font-weight: 500;
      z-index: 1000000;
      box-shadow: 0 6px 20px rgba(0,0,0,0.3);
      animation: slideIn 0.3s ease-out;
      max-width: 400px;
      min-width: 320px;
    `;

    // 添加内部样式（如果还没有）
    if (!document.querySelector('#wzslinker-detailed-error-styles')) {
      const style = document.createElement('style');
      style.id = 'wzslinker-detailed-error-styles';
      style.textContent = `
        .wzslinker-service-error .error-header {
          display: flex;
          align-items: center;
          justify-content: space-between;
          padding: 12px 16px;
          color: white;
          border-radius: 6px 6px 0 0;
          font-weight: bold;
        }
        .wzslinker-service-error .error-icon {
          font-size: 18px;
          margin-right: 8px;
        }
        .wzslinker-service-error .error-close {
          background: rgba(255,255,255,0.2);
          border: none;
          font-size: 18px;
          font-weight: bold;
          color: white;
          cursor: pointer;
          padding: 0;
          width: 24px;
          height: 24px;
          display: flex;
          align-items: center;
          justify-content: center;
          border-radius: 50%;
          transition: background-color 0.2s;
        }
        .wzslinker-service-error .error-close:hover {
          background: rgba(255,255,255,0.3);
        }
        .wzslinker-service-error .error-content {
          padding: 16px;
        }
        .wzslinker-service-error .error-content p {
          margin: 0 0 12px 0;
          line-height: 1.5;
        }
        .wzslinker-service-error .error-suggestion {
          background: #f5f5f5;
          padding: 8px;
          border-radius: 4px;
          font-size: 13px;
        }
        .wzslinker-service-error .error-buttons {
          display: flex;
          gap: 8px;
          margin-top: 12px;
        }
        .wzslinker-service-error .error-button {
          flex: 1;
          padding: 8px 12px;
          border: none;
          border-radius: 4px;
          font-size: 13px;
          font-weight: bold;
          cursor: pointer;
          transition: all 0.2s;
        }
        .wzslinker-service-error .error-button-primary {
          background: #4a90e2;
          color: white;
        }
        .wzslinker-service-error .error-button-primary:hover {
          background: #357abd;
        }
        .wzslinker-service-error .error-button-secondary {
          background: #f5f5f5;
          color: #333;
        }
        .wzslinker-service-error .error-button-secondary:hover {
          background: #e0e0e0;
        }
      `;
      document.head.appendChild(style);
    }

    // 添加事件监听器
    const closeButton = notification.querySelector('.error-close');
    const primaryButton = notification.querySelector('.error-button-primary');
    const diagnosticBtn = notification.querySelector('#diagnosticBtn');
    
    if (closeButton) {
      closeButton.addEventListener('click', () => {
        notification.remove();
      });
    }
    
    if (primaryButton) {
      primaryButton.addEventListener('click', () => {
        window.open('https://hrfocus.top', '_blank');
      });
    }
    
    if (diagnosticBtn) {
      diagnosticBtn.addEventListener('click', async () => {
        await this.runDiagnostic();
      });
    }

    document.body.appendChild(notification);

    // 10秒后自动移除
    setTimeout(() => {
      if (notification.parentNode) {
        notification.style.animation = 'slideOut 0.3s ease-in';
        setTimeout(() => {
          if (notification.parentNode) {
            notification.remove();
          }
        }, 300);
      }
    }, 10000);
  }

  // 显示服务错误通知（简化版，向后兼容）
  showServiceErrorNotification(message, suggestion) {
    this.showDetailedErrorNotification('UNKNOWN', message, suggestion || '请检查本地服务是否正在运行');
  }

  // 运行诊断
  async runDiagnostic() {
    try {
      console.log('WzSLinker: 开始运行网络诊断...');
      
      // 显示诊断中提示
      this.showNotification('正在运行诊断，请稍候...', 'info');
      
      // 调用background的诊断功能
      const response = await chrome.runtime.sendMessage({ 
        action: 'diagnoseConnection' 
      });
      
      if (response && response.success) {
        const results = response.data;
        console.log('诊断结果:', results);
        
        // 显示诊断结果
        this.showDiagnosticResults(results);
      }
    } catch (error) {
      console.error('运行诊断失败:', error);
      this.showNotification('诊断失败，请查看控制台', 'error');
    }
  }

  // 显示诊断结果
  showDiagnosticResults(results) {
    // 移除现有通知
    const existing = document.querySelector('.wzslinker-diagnostic-results');
    if (existing) {
      existing.remove();
    }

    const resultsHtml = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="UTF-8">
        <title>WzSLinker 诊断报告</title>
        <style>
          body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            padding: 20px; 
            background: #f5f5f5;
            margin: 0;
          }
          .diagnostic-results {
            background: white;
            padding: 24px;
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.1);
            max-width: 700px;
            margin: 0 auto;
          }
          .diagnostic-results h2 {
            margin-top: 0;
            color: #4a90e2;
            border-bottom: 3px solid #4a90e2;
            padding-bottom: 12px;
          }
          .diagnostic-info {
            background: #e3f2fd;
            padding: 12px;
            border-radius: 6px;
            margin-bottom: 20px;
            font-size: 14px;
          }
          .diagnostic-results ul {
            list-style: none;
            padding: 0;
          }
          .diagnostic-results li {
            margin: 15px 0;
            padding: 15px;
            border-radius: 6px;
            background: #f8f9fa;
          }
          .diagnostic-results li.success {
            border-left: 5px solid #4caf50;
            background: #e8f5e9;
          }
          .diagnostic-results li.failed {
            border-left: 5px solid #f44336;
            background: #ffebee;
          }
          .diagnostic-results li.info {
            border-left: 5px solid #2196f3;
            background: #e3f2fd;
          }
          .diagnostic-results .test-name {
            font-weight: bold;
            font-size: 16px;
            margin-bottom: 8px;
          }
          .diagnostic-results .test-message {
            color: #555;
            margin-bottom: 4px;
          }
          .diagnostic-results .test-error {
            color: #d32f2f;
            font-size: 13px;
            font-family: monospace;
            background: rgba(0,0,0,0.05);
            padding: 4px 8px;
            border-radius: 3px;
            margin-top: 6px;
          }
          .diagnostic-results .test-suggestion {
            color: #1976d2;
            font-size: 13px;
            margin-top: 6px;
            font-style: italic;
          }
        </style>
      </head>
      <body>
        <div class="diagnostic-results">
          <h2>🔍 WzSLinker 诊断报告</h2>
          <div class="diagnostic-info">
            <strong>诊断时间：</strong> ${new Date(results.timestamp).toLocaleString()}<br>
            <strong>服务地址：</strong> ${results.host}:${results.port}
          </div>
          <ul>
            ${results.tests.map(test => `
              <li class="${test.status.toLowerCase()}">
                <div class="test-name">📋 ${test.name}</div>
                <div class="test-message">${test.message}</div>
                ${test.error ? `<div class="test-error">错误详情: ${test.error}</div>` : ''}
                ${test.suggestion ? `<div class="test-suggestion">💡 ${test.suggestion}</div>` : ''}
              </li>
            `).join('')}
          </ul>
        </div>
      </body>
      </html>
    `;
    
    // 在新窗口中显示
    const win = window.open('', '_blank', 'width=750,height=600,scrollbars=yes');
    if (win) {
      win.document.write(resultsHtml);
      win.document.close();
    } else {
      // 如果弹窗被阻止，在当前页面显示
      this.showNotification('诊断完成，请查看控制台', 'success');
      console.log('诊断结果:', results);
    }
  }

  /**
   * ==================== 价格指示器功能 ====================
   */

  /**
   * 添加价格指示器（优化版 - 支持批量请求）
   */
  addPriceIndicator(element, stockCode) {
    // 防止重复创建
    const existingIndicator = element.querySelector('.stock-price-indicator');
    if (existingIndicator) {
      this.updateStockPrice(stockCode, existingIndicator);
      return;
    }

    // 优化：先检查缓存，有数据再创建DOM
    const cached = this.priceCache.get(stockCode);
    
    // 创建指示器
    const indicator = document.createElement('div');
    indicator.className = cached ? 'stock-price-indicator' : 'stock-price-indicator loading';
    indicator.setAttribute('data-stock-code', stockCode);
    indicator.textContent = cached ? '' : '...';
    
    // 设置位置为相对于父元素
    element.style.position = 'relative';
    element.appendChild(indicator);
    
    // 如果有缓存，立即显示；否则加入批量队列
    if (cached && Date.now() - cached.timestamp < this.cacheTimeout) {
      this.updateIndicatorDisplay(indicator, cached.data);
    } else {
      // 新增：加入批量请求队列（提升性能）
      this.addToBatchQueue(stockCode, indicator);
    }
    
    // 点击刷新功能已禁用（用户要求不可点击）
    // indicator.addEventListener('click', ...) 已移除
  }

  /**
   * 加入批量请求队列（新增：优化100+股票场景）
   */
  addToBatchQueue(stockCode, indicator) {
    // 添加到批量队列
    this.batchQueue.push({ stockCode, indicator });
    
    // 清除之前的定时器
    if (this.batchTimeout) {
      clearTimeout(this.batchTimeout);
    }
    
    // 设置新的定时器，100ms后执行批量请求
    this.batchTimeout = setTimeout(() => {
      this.processBatchQueue();
    }, 100);
    
    // 如果队列达到批量大小，立即执行
    if (this.batchQueue.length >= this.batchSize) {
      clearTimeout(this.batchTimeout);
      this.processBatchQueue();
    }
  }

  /**
   * 处理批量队列（新增：批量获取股票数据）
   */
  async processBatchQueue() {
    if (this.batchQueue.length === 0) return;
    
    // 取出当前批次
    const batch = this.batchQueue.splice(0, this.batchSize);
    const stockCodes = batch.map(item => item.stockCode);
    
    console.log(`WzSLinker: 批量请求 ${stockCodes.length} 个股票`, stockCodes);
    
    try {
      // 批量获取数据
      const results = await this.fetchStockPriceBatch(stockCodes);
      
      // 更新每个指示器
      batch.forEach(({ stockCode, indicator }) => {
        const data = results.get(stockCode);
        if (data && indicator.isConnected) {
          // 缓存数据
          this.priceCache.set(stockCode, {
            data: data,
            timestamp: Date.now()
          });
          // 更新显示
          this.updateIndicatorDisplay(indicator, data);
        } else if (indicator.isConnected) {
          // 批量请求失败，降级为单个请求
          this.updateStockPrice(stockCode, indicator);
        }
      });
      
    } catch (error) {
      console.error('WzSLinker: 批量请求失败，降级为单个请求', error);
      // 降级：逐个请求
      batch.forEach(({ stockCode, indicator }) => {
        this.updateStockPrice(stockCode, indicator);
      });
    }
    
    // 如果还有剩余，继续处理
    if (this.batchQueue.length > 0) {
      setTimeout(() => this.processBatchQueue(), 50);
    }
  }

  /**
   * 更新股票价格
   */
  async updateStockPrice(stockCode, indicator) {
    // 检查DOM是否有效
    if (!indicator || !indicator.isConnected) {
      return;
    }

    // 检查缓存
    const cached = this.priceCache.get(stockCode);
    if (cached && Date.now() - cached.timestamp < this.cacheTimeout) {
      this.updateIndicatorDisplay(indicator, cached.data);
      return;
    }

    // 防抖去重
    const updateKey = `update_${stockCode}`;
    if (this.requestQueue.has(updateKey)) {
      return;
    }

    try {
      this.requestQueue.set(updateKey, true);
      
      // 等待请求槽位
      await this.waitForRequestSlot();
      
      // 标记为加载中
      indicator.className = 'stock-price-indicator loading';
      indicator.textContent = '...';
      
      // 获取价格数据
      const data = await this.fetchStockPrice(stockCode);
      
      // 缓存数据
      this.priceCache.set(stockCode, {
        data: data,
        timestamp: Date.now()
      });
      
      // 更新显示
      this.updateIndicatorDisplay(indicator, data);
      
    } catch (error) {
      console.error(`WzSLinker: 获取股票${stockCode}价格失败`, error);
      indicator.className = 'stock-price-indicator error';
      indicator.textContent = '?';
      indicator.title = '获取价格失败，点击重试';
    } finally {
      this.requestQueue.delete(updateKey);
      this.requestInProgress.delete(stockCode);
    }
  }

  /**
   * 等待请求槽位
   */
  async waitForRequestSlot() {
    // 频率限制
    const timeSinceLastRequest = Date.now() - this.lastRequestTime;
    if (timeSinceLastRequest < this.requestRateLimit) {
      await new Promise(resolve => 
        setTimeout(resolve, this.requestRateLimit - timeSinceLastRequest)
      );
    }
    
    // 并发控制
    while (this.requestInProgress.size >= this.maxConcurrentRequests) {
      await new Promise(resolve => setTimeout(resolve, 100));
    }
    
    this.lastRequestTime = Date.now();
  }

  /**
   * 获取股票价格（多数据源，优化版 - 并行请求+超时控制）
   */
  async fetchStockPrice(stockCode) {
    this.requestInProgress.add(stockCode);
    
    try {
      // 优化策略：并行请求多个数据源，谁先返回用谁的
      const fetchPromises = [
        this.fetchWithTimeout(() => this.fetchFromEastMoney(stockCode), 3000, 'eastmoney'),
        this.fetchWithTimeout(() => this.fetchFromSina(stockCode), 2000, 'sina'),
        this.fetchWithTimeout(() => this.fetchFromTencent(stockCode), 2000, 'tencent')
      ];

      // 使用Promise.race获取最快响应的数据源
      const result = await Promise.race(
        fetchPromises.map(p => p.then(r => {
          if (r && r.currentPrice > 0) {
            return r;
          }
          throw new Error('数据无效');
        }).catch(err => {
          // 继续等待其他请求
          return new Promise((resolve, reject) => {
            setTimeout(() => reject(err), 5000);
          });
        }))
      );

      if (result && result.currentPrice > 0) {
        console.log(`WzSLinker: 使用 ${result.source} 数据源，响应快速`);
        return result;
      }

      // 如果并行请求全部失败，尝试网易作为最后备用
      try {
        const neteaseResult = await this.fetchFromNetease(stockCode);
        if (neteaseResult && neteaseResult.currentPrice > 0) {
          return neteaseResult;
        }
      } catch (error) {
        console.warn('WzSLinker: 网易财经API失败', error);
      }
      
      throw new Error('所有数据源均失败');
    } catch (error) {
      console.error(`WzSLinker: 获取股票${stockCode}价格失败`, error);
      throw error;
    } finally {
      this.requestInProgress.delete(stockCode);
    }
  }

  /**
   * 带超时控制的请求（优化：避免慢速请求阻塞）
   */
  async fetchWithTimeout(fetchFn, timeout, sourceName) {
    return Promise.race([
      fetchFn(),
      new Promise((_, reject) => 
        setTimeout(() => reject(new Error(`${sourceName} 请求超时`)), timeout)
      )
    ]);
  }

  /**
   * 批量获取股票价格（新增：支持一次请求多个股票）
   */
  async fetchStockPriceBatch(stockCodes) {
    const results = new Map();
    
    try {
      // 尝试使用东方财富批量API
      const batchResults = await this.fetchFromEastMoneyBatch(stockCodes);
      batchResults.forEach((data, code) => {
        if (data && data.currentPrice > 0) {
          results.set(code, data);
        }
      });
      
      // 如果批量请求成功获取了所有数据，直接返回
      if (results.size === stockCodes.length) {
        console.log(`WzSLinker: 批量请求成功，获取了 ${results.size} 个股票数据`);
        return results;
      }
      
      // 对于批量请求失败的股票，并行单独请求
      const failedCodes = stockCodes.filter(code => !results.has(code));
      if (failedCodes.length > 0) {
        console.log(`WzSLinker: ${failedCodes.length} 个股票批量请求失败，尝试单独请求`);
        
        const individualPromises = failedCodes.map(async (code) => {
          try {
            const data = await this.fetchStockPrice(code);
            if (data && data.currentPrice > 0) {
              results.set(code, data);
            }
          } catch (error) {
            console.warn(`WzSLinker: 股票${code}单独请求也失败`, error);
          }
        });
        
        await Promise.all(individualPromises);
      }
      
    } catch (error) {
      console.error('WzSLinker: 批量请求完全失败，降级为并行单独请求', error);
      
      // 完全降级：并行单独请求所有股票
      const promises = stockCodes.map(async (code) => {
        try {
          const data = await this.fetchStockPrice(code);
          if (data && data.currentPrice > 0) {
            results.set(code, data);
          }
        } catch (error) {
          console.warn(`WzSLinker: 股票${code}请求失败`, error);
        }
      });
      
      await Promise.all(promises);
    }
    
    return results;
  }

  /**
   * 从东方财富批量获取数据（新增：支持批量查询）
   */
  async fetchFromEastMoneyBatch(stockCodes) {
    try {
      // 构建批量请求的secid列表
      const secids = stockCodes.map(code => 
        code.startsWith('6') ? `1.${code}` : `0.${code}`
      ).join(',');
      
      // 使用批量API（东方财富支持用逗号分隔多个股票）
      const url = `https://push2.eastmoney.com/api/qt/ulist.np/get?ut=fa5fd1943c7b386f172d6893dbfba10b&fltt=2&fields=f12,f14,f2,f3,f15,f16,f17,f18&secids=${secids}`;
      
      const response = await fetch(url, {
        method: 'GET',
        mode: 'cors',
        credentials: 'omit'
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const data = await response.json();
      return this.parseEastMoneyBatchData(data, stockCodes);
    } catch (error) {
      console.error('WzSLinker: 东方财富批量API失败', error);
      throw error;
    }
  }

  /**
   * 解析东方财富批量数据（新增）
   */
  parseEastMoneyBatchData(data, stockCodes) {
    const results = new Map();
    
    try {
      if (!data.data || !data.data.diff) {
        console.error('WzSLinker: 东方财富批量API返回数据格式错误');
        throw new Error('批量数据格式错误');
      }

      const stockList = data.data.diff;
      
      // 解析每个股票数据
      stockList.forEach(stockInfo => {
        const stockCode = stockInfo.f12; // 股票代码
        const name = stockInfo.f14 || '未知';
        const currentPrice = parseFloat(stockInfo.f2) || 0; // 当前价
        const changePercent = parseFloat(stockInfo.f3) || 0; // 涨跌幅
        
        // 通过涨跌幅和当前价计算昨收价和涨跌额
        const previousClose = changePercent !== 0 
          ? currentPrice / (1 + changePercent / 100) 
          : currentPrice;
        const change = currentPrice - previousClose;
        
        if (currentPrice > 0 && stockCode) {
          results.set(stockCode, {
            name,
            currentPrice,
            previousClose,
            change,
            changePercent,
            stockCode,
            source: 'eastmoney-batch'
          });
        }
      });
      
      console.log(`WzSLinker: 东方财富批量API解析成功，获取了 ${results.size}/${stockCodes.length} 个股票数据`);
      
    } catch (error) {
      console.error('WzSLinker: 解析东方财富批量数据失败', error);
      throw error;
    }
    
    return results;
  }

  /**
   * 从东方财富获取数据（优化版 - 使用移动端UA加速）
   */
  async fetchFromEastMoney(stockCode) {
    try {
      const secid = stockCode.startsWith('6') ? `1.${stockCode}` : `0.${stockCode}`;
      // 优化：简化请求字段，只获取必要数据以提高速度
      const url = `https://push2.eastmoney.com/api/qt/stock/get?ut=fa5fd1943c7b386f172d6893dbfba10b&invt=2&fltt=2&fields=f43,f58,f60&secid=${secid}`;
      
      const response = await fetch(url, {
        method: 'GET',
        mode: 'cors',
        credentials: 'omit'
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const data = await response.json();
      return this.parseEastMoneyData(data, stockCode);
    } catch (error) {
      console.error('WzSLinker: 东方财富API失败', error);
      throw error;
    }
  }

  /**
   * 解析东方财富数据
   */
  parseEastMoneyData(data, stockCode) {
    try {
      if (!data.data) {
        console.error('WzSLinker: 东方财富API返回数据:', JSON.stringify(data));
        throw new Error('东方财富数据格式错误');
      }

      const stockInfo = data.data;
      console.log(`WzSLinker: 东方财富API解析数据:`, stockInfo);
      
      const name = stockInfo.f58 || '未知';
      const currentPrice = parseFloat(stockInfo.f43) || 0;
      const previousClose = parseFloat(stockInfo.f60) || 0;
      const change = currentPrice - previousClose;
      const changePercent = previousClose > 0 ? (change / previousClose) * 100 : 0;

      if (currentPrice === 0 || previousClose === 0) {
        console.warn(`WzSLinker: 东方财富API数据无效 - 当前价:${currentPrice}, 昨收:${previousClose}`);
        throw new Error('价格数据无效');
      }

      const result = {
        name,
        currentPrice,
        previousClose,
        change,
        changePercent,
        stockCode,
        source: 'eastmoney'
      };
      
      console.log(`WzSLinker: 东方财富API解析成功:`, result);
      return result;
    } catch (error) {
      console.error('WzSLinker: 解析东方财富数据失败', error);
      throw error;
    }
  }

  /**
   * 从新浪财经获取数据（优化版 - 使用Edge UA）
   */
  async fetchFromSina(stockCode) {
    try {
      const prefix = stockCode.startsWith('6') ? 'sh' : 'sz';
      const url = `https://hq.sinajs.cn/list=${prefix}${stockCode}`;
      
      const response = await fetch(url, {
        method: 'GET',
        mode: 'cors',
        credentials: 'omit'
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }

      const text = await response.text();
      return this.parseSinaData(text, stockCode);
    } catch (error) {
      console.error('WzSLinker: 新浪财经API失败', error);
      throw error;
    }
  }

  /**
   * 解析新浪财经数据
   */
  parseSinaData(text, stockCode) {
    const match = text.match(/"([^"]+)"/);
    if (!match) throw new Error('新浪数据格式错误');
    
    const parts = match[1].split(',');
    const name = parts[0];
    const currentPrice = parseFloat(parts[3]) || 0;
    const previousClose = parseFloat(parts[2]) || 0;
    const change = currentPrice - previousClose;
    const changePercent = previousClose > 0 ? (change / previousClose) * 100 : 0;

    return {
      name,
      currentPrice,
      previousClose,
      change,
      changePercent,
      stockCode,
      source: 'sina'
    };
  }

  /**
   * 从腾讯财经获取数据（优化版 - 使用Chrome UA）
   */
  async fetchFromTencent(stockCode) {
    try {
      const prefix = stockCode.startsWith('6') ? 'sh' : 'sz';
      const url = `https://qt.gtimg.cn/q=${prefix}${stockCode}`;
      
      const response = await fetch(url, {
        method: 'GET',
        mode: 'cors',
        credentials: 'omit'
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }

      const text = await response.text();
      return this.parseTencentData(text, stockCode);
    } catch (error) {
      console.error('WzSLinker: 腾讯财经API失败', error);
      throw error;
    }
  }

  /**
   * 解析腾讯财经数据
   */
  parseTencentData(text, stockCode) {
    const match = text.match(/="([^"]+)"/);
    if (!match) throw new Error('腾讯数据格式错误');
    
    const parts = match[1].split('~');
    const name = parts[1];
    const currentPrice = parseFloat(parts[3]) || 0;
    const previousClose = parseFloat(parts[4]) || 0;
    const change = currentPrice - previousClose;
    const changePercent = previousClose > 0 ? (change / previousClose) * 100 : 0;

    return {
      name,
      currentPrice,
      previousClose,
      change,
      changePercent,
      stockCode,
      source: 'tencent'
    };
  }

  /**
   * 从网易财经获取数据（优化版 - 使用移动端UA）
   */
  async fetchFromNetease(stockCode) {
    try {
      const prefix = stockCode.startsWith('6') ? 'sh' : 'sz';
      const url = `http://api.money.163.com/api/security/quotes/${prefix}${stockCode}.html`;
      
      const response = await fetch(url, {
        method: 'GET',
        mode: 'cors',
        credentials: 'omit'
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }

      const data = await response.json();
      return this.parseNeteaseData(data, stockCode);
    } catch (error) {
      console.error('WzSLinker: 网易财经API失败', error);
      throw error;
    }
  }

  /**
   * 解析网易财经数据
   */
  parseNeteaseData(data, stockCode) {
    if (!data || !data.data || !data.data.length) {
      throw new Error('网易数据格式错误');
    }
    
    const stockInfo = data.data[0];
    const name = stockInfo.name;
    const currentPrice = parseFloat(stockInfo.price) || 0;
    const previousClose = parseFloat(stockInfo.yestclose) || 0;
    const change = currentPrice - previousClose;
    const changePercent = previousClose > 0 ? (change / previousClose) * 100 : 0;

    return {
      name,
      currentPrice,
      previousClose,
      change,
      changePercent,
      stockCode,
      source: 'netease'
    };
  }

  /**
   * 更新指示器显示
   */
  updateIndicatorDisplay(indicator, data) {
    if (!indicator || !indicator.isConnected) {
      return;
    }

    const { changePercent } = data;
    
    // 验证数据
    if (typeof changePercent !== 'number' || isNaN(changePercent)) {
      indicator.textContent = '?';
      return;
    }
    
    // 确定颜色和CSS类
    let className = 'stock-price-indicator';
    let color;
    
    if (changePercent > 0) {
      className += ' positive';
      color = '#ff69b4'; // 粉色(上涨)
    } else if (changePercent < 0) {
      className += ' negative';
      color = '#00ff00'; // 绿色(下跌)
    } else {
      className += ' neutral';
      color = '#666666'; // 灰色(平盘)
    }

    indicator.className = className;
    
    // 显示格式化的涨跌幅
    const sign = changePercent > 0 ? '+' : '';
    indicator.textContent = `${sign}${changePercent.toFixed(2)}%`;
    
    // 应用样式（完全透明，无背景）
    indicator.style.cssText = `
      position: absolute !important;
      top: -7px !important;
      right: -12px !important;
      color: ${color} !important;
      font-size: 10px !important;
      font-weight: bold !important;
      z-index: 2147483647 !important;
      background: none !important;
      background-color: transparent !important;
      padding: 0 !important;
      margin: 0 !important;
      border: none !important;
      border-radius: 0 !important;
      white-space: nowrap !important;
      cursor: default !important;
      box-shadow: none !important;
      pointer-events: none !important;
      outline: none !important;
    `;
    
    // 悬停提示（简化，因为不可点击）
    indicator.title = `当前价: ${data.currentPrice.toFixed(2)}\n涨跌: ${data.change.toFixed(2)}\n涨跌幅: ${sign}${changePercent.toFixed(2)}%\n数据来源: ${data.source}`;
  }

  /**
   * 启动定时更新
   */
  startPriceUpdates() {
    // 清除现有定时器
    if (this.priceUpdateInterval) {
      clearInterval(this.priceUpdateInterval);
    }

    // 每60秒更新一次所有价格
    this.priceUpdateInterval = setInterval(() => {
      this.updateAllStockPrices();
    }, 60000);
  }

  /**
   * 更新所有股票价格
   */
  updateAllStockPrices() {
    const indicators = document.querySelectorAll('.stock-price-indicator');
    
    indicators.forEach(indicator => {
      const stockCode = indicator.getAttribute('data-stock-code');
      if (stockCode) {
        this.updateStockPrice(stockCode, indicator);
      }
    });
  }

  /**
   * 恢复价格指示器（页面刷新后或开关打开时）
   */
  restorePriceIndicators() {
    if (!this.settings.showPriceEnabled) {
      return;
    }

    console.log('WzSLinker: 开始恢复价格指示器');

    // 查找所有高亮元素
    const highlights = document.querySelectorAll('stock-highlight[data-stock-code]');
    
    highlights.forEach(highlight => {
      const stockCode = highlight.getAttribute('data-stock-code');
      const existingIndicator = highlight.querySelector('.stock-price-indicator');
      
      // 如果没有价格指示器，重新创建
      if (stockCode && !existingIndicator) {
        setTimeout(() => {
          this.addPriceIndicator(highlight, stockCode);
        }, 100);
      }
    });
    
    // 重新启动定时更新
    this.startPriceUpdates();
    
    console.log(`WzSLinker: 已为 ${highlights.length} 个股票恢复价格指示器`);
  }

  /**
   * 移除所有价格指示器
   */
  removeAllPriceIndicators() {
    console.log('WzSLinker: 开始移除所有价格指示器');
    
    // 查找并移除所有价格指示器
    const indicators = document.querySelectorAll('.stock-price-indicator');
    indicators.forEach(indicator => {
      if (indicator.parentNode) {
        indicator.remove();
      }
    });
    
    // 清除价格缓存
    this.priceCache.clear();
    
    // 停止定时更新
    if (this.priceUpdateInterval) {
      clearInterval(this.priceUpdateInterval);
      this.priceUpdateInterval = null;
    }
    
    console.log(`WzSLinker: 已移除 ${indicators.length} 个价格指示器`);
  }

  /**
   * ==================== 域名管理功能 ====================
   */

  /**
   * 获取当前页面域名或文件路径
   * @returns {string} 当前域名或文件路径
   */
  getCurrentDomain() {
    try {
      // 如果是 file:// 协议，返回完整的文件路径
      if (window.location.protocol === 'file:') {
        // 标准化文件路径
        let filePath = window.location.href;
        // 统一使用正斜杠
        filePath = filePath.replace(/\\/g, '/');
        return filePath;
      }
      
      let hostname = window.location.hostname;
      
      // 如果有端口，保留端口
      if (window.location.port) {
        hostname += ':' + window.location.port;
      }
      
      return hostname;
    } catch (error) {
      console.error('WzSLinker: 获取当前域名失败', error);
      return '';
    }
  }

  /**
   * 检查当前域名是否被允许运行
   * @returns {boolean} true表示允许运行，false表示禁止运行
   */
  isDomainAllowed() {
    const currentDomain = this.getCurrentDomain();
    if (!currentDomain) return true; // 如果无法获取域名，默认允许
    
    // 屏蔽模式：如果当前域名在黑名单中，禁止运行
    if (this.settings.domainBlocklistEnabled && this.settings.domainBlocklist) {
      for (const blockedDomain of this.settings.domainBlocklist) {
        if (this.matchDomain(currentDomain, blockedDomain)) {
          console.log(`WzSLinker: 当前域名 ${currentDomain} 在黑名单中，功能已禁用`);
          return false;
        }
      }
    }
    
    // 允许模式：如果启用且当前域名不在白名单中，禁止运行
    if (this.settings.domainAllowlistEnabled && this.settings.domainAllowlist) {
      let isAllowed = false;
      for (const allowedDomain of this.settings.domainAllowlist) {
        if (this.matchDomain(currentDomain, allowedDomain)) {
          isAllowed = true;
          break;
        }
      }
      
      if (!isAllowed) {
        console.log(`WzSLinker: 当前域名 ${currentDomain} 不在白名单中，功能已禁用`);
        return false;
      }
    }
    
    return true; // 默认允许
  }

  /**
   * 匹配域名或文件路径（支持精确匹配、子域名匹配、文件路径匹配）
   * @param {string} currentDomain - 当前域名或文件路径
   * @param {string} targetDomain - 目标域名或文件路径
   * @returns {boolean} 是否匹配
   */
  matchDomain(currentDomain, targetDomain) {
    if (!currentDomain || !targetDomain) return false;
    
    // 如果都是 file:// 协议，进行文件路径匹配
    if (currentDomain.startsWith('file:///') && targetDomain.startsWith('file:///')) {
      // 完全匹配
      if (currentDomain === targetDomain) return true;
      
      // URL解码后再匹配（处理中文路径）
      try {
        const currentDecoded = decodeURIComponent(currentDomain);
        const targetDecoded = decodeURIComponent(targetDomain);
        if (currentDecoded === targetDecoded) return true;
      } catch (e) {
        // 解码失败，继续其他匹配
      }
      
      return false;
    }
    
    // 如果一个是 file://，另一个不是，不匹配
    if (currentDomain.startsWith('file:///') || targetDomain.startsWith('file:///')) {
      return false;
    }
    
    // 完全匹配
    if (currentDomain === targetDomain) return true;
    
    // 子域名匹配：www.example.com 匹配 example.com
    if (currentDomain.endsWith('.' + targetDomain)) return true;
    
    // 反向匹配：example.com 匹配 www.example.com
    if (targetDomain.endsWith('.' + currentDomain)) return true;
    
    return false;
  }

}

// 初始化内容脚本
console.log('WzSLinker: Content script 开始加载...');
let wzsLinker;

function initializeWzSLinker() {
  console.log('WzSLinker: 初始化WzSLinker实例...');
  wzsLinker = new WzSLinkerContentScript();
  // 将实例暴露到全局作用域，方便测试
  window.wzsLinker = wzsLinker;
  console.log('WzSLinker: 实例已创建并暴露到window.wzsLinker');
  
  // 添加调试信息
  console.log('WzSLinker: 当前页面URL:', window.location.href);
  console.log('WzSLinker: 页面标题:', document.title);
  console.log('WzSLinker: 实例方法:', Object.getOwnPropertyNames(Object.getPrototypeOf(wzsLinker)));
}

if (document.readyState === 'loading') {
  console.log('WzSLinker: 页面正在加载，等待DOMContentLoaded事件...');
  document.addEventListener('DOMContentLoaded', initializeWzSLinker);
} else {
  console.log('WzSLinker: 页面已加载完成，立即初始化...');
  initializeWzSLinker();
}

// 添加全局测试函数
window.testWzSLinkerNotification = function() {
  console.log('WzSLinker: 开始测试通知功能');
  if (window.wzsLinker) {
    window.wzsLinker.showServiceErrorNotification('这是一个测试通知，用于验证页面内通知功能是否正常工作。');
    console.log('WzSLinker: 测试通知已触发');
  } else {
    console.error('WzSLinker: 未找到WzSLinker实例');
  }
};

// 添加价格测试函数
window.testWzSLinkerStockPrice = async function(stockCode) {
  console.log(`WzSLinker: 测试获取股票 ${stockCode} 的价格`);
  if (window.wzsLinker) {
    try {
      const data = await window.wzsLinker.fetchStockPrice(stockCode);
      console.log('WzSLinker: 价格数据获取成功:', data);
      return data;
    } catch (error) {
      console.error('WzSLinker: 价格数据获取失败:', error);
      return null;
    }
  } else {
    console.error('WzSLinker: 未找到WzSLinker实例');
    return null;
  }
};
