// WzSLinker 设置页面逻辑

class OptionsManager {
  constructor() {
    this.settings = {};
    this.init();
  }

  async init() {
    await this.loadSettings();
    this.setupEventListeners();
    this.updateUI();
  }

  // 加载设置
  async loadSettings() {
    try {
      const result = await chrome.storage.local.get([
        'highlightStyle',
        'highlightColor',
        'textColor',
        'highlightEnabled',
        'stockData',
        'localServiceHost',
        'localServicePort',
        'requestTimeout',
        'retryEnabled',
        'maxRetries',
        'linkHistory'
      ]);
      
      this.settings = {
        highlightStyle: result.highlightStyle || 'none',
        highlightColor: result.highlightColor || '#fff3cd',
        textColor: result.textColor || '#ff0000',
        highlightEnabled: result.highlightEnabled !== undefined ? result.highlightEnabled : true,
        stockData: result.stockData || {},
        localServiceHost: result.localServiceHost || '127.0.0.1',
        localServicePort: result.localServicePort || '3000',
        requestTimeout: result.requestTimeout || 5000,
        retryEnabled: result.retryEnabled !== false,
        maxRetries: result.maxRetries || 2,
        linkHistory: result.linkHistory || []
      };
    } catch (error) {
      console.error('加载设置失败:', error);
      this.settings = this.getDefaultSettings();
    }
  }

  // 获取默认设置（使用公共工具函数）
  getDefaultSettings() {
    return window.WzSLinkerUtils ? window.WzSLinkerUtils.getDefaultSettings() : this._getDefaultSettings();
  }

  // 备用默认设置函数
  _getDefaultSettings() {
    return {
      highlightStyle: 'none',
      highlightColor: '#fff3cd',
      textColor: '#ff0000',
      highlightEnabled: true,
      stockData: {},
      localServiceHost: '127.0.0.1',
      localServicePort: '3000',
      requestTimeout: 5000,    // 新增：请求超时时间（毫秒）
      retryEnabled: true,      // 新增：启用重试
      maxRetries: 2,           // 新增：最大重试次数
      linkHistory: []
    };
  }

  // 设置事件监听器
  setupEventListeners() {
    // 保存设置按钮
    const saveSettingsButton = document.getElementById('saveSettingsButton');
    if (saveSettingsButton) {
      saveSettingsButton.addEventListener('click', () => {
        this.saveSettings();
      });
    }

    // 导入数据按钮
    const importDataButton = document.getElementById('importDataButton');
    if (importDataButton) {
      importDataButton.addEventListener('click', () => {
        this.importStockData();
      });
    }

    // 导出数据按钮
    const exportDataButton = document.getElementById('exportDataButton');
    if (exportDataButton) {
      exportDataButton.addEventListener('click', () => {
        this.exportStockData();
      });
    }

    // 添加股票按钮
    const addStockButtonOpt = document.getElementById('addStockButtonOpt');
    if (addStockButtonOpt) {
      addStockButtonOpt.addEventListener('click', () => {
        this.addSingleStock();
      });
    }

    // 清空所有股票数据按钮
    const clearAllStocksButton = document.getElementById('clearAllStocksButton');
    if (clearAllStocksButton) {
      clearAllStocksButton.addEventListener('click', () => {
        this.clearAllStockData();
      });
    }

    // 清空联动记录按钮
    const clearLinkedHistoryButton = document.getElementById('clearLinkedHistoryButton');
    if (clearLinkedHistoryButton) {
      clearLinkedHistoryButton.addEventListener('click', () => {
        this.clearLinkedHistory();
      });
    }

    // 重置设置按钮
    const resetSettingsButton = document.getElementById('resetSettingsButton');
    if (resetSettingsButton) {
      resetSettingsButton.addEventListener('click', () => {
        this.resetSettings();
      });
    }

    // 文件上传
    const stockDataUpload = document.getElementById('stockDataUpload');
    if (stockDataUpload) {
      stockDataUpload.addEventListener('change', (e) => {
        this.handleFileUpload(e.target.files[0]);
      });
    }

    // 回车键添加股票
    const addStockCodeOpt = document.getElementById('addStockCodeOpt');
    if (addStockCodeOpt) {
      addStockCodeOpt.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
          this.addSingleStock();
        }
      });
    }

    // 高亮开关
    const highlightEnabled = document.getElementById('highlightEnabled');
    if (highlightEnabled) {
      highlightEnabled.addEventListener('change', () => {
        this.updateColorPreview();
        // 立即保存设置并通知content script
        this.saveHighlightEnabled();
      });
    }

    // 颜色设置变化时更新预览
    const highlightColor = document.getElementById('highlightColor');
    if (highlightColor) {
      highlightColor.addEventListener('input', () => {
        this.updateColorPreview();
      });
    }

    const textColor = document.getElementById('textColor');
    if (textColor) {
      textColor.addEventListener('input', () => {
        this.updateColorPreview();
      });
    }

    const highlightStyle = document.getElementById('highlightStyle');
    if (highlightStyle) {
      highlightStyle.addEventListener('change', () => {
        this.updateColorPreview();
      });
    }

    // 测试连接按钮
    const testConnectionButton = document.getElementById('testConnectionButton');
    if (testConnectionButton) {
      testConnectionButton.addEventListener('click', () => {
        this.testConnection();
      });
    }
  }

  // 更新UI
  updateUI() {
    const highlightEnabled = document.getElementById('highlightEnabled');
    if (highlightEnabled) highlightEnabled.checked = this.settings.highlightEnabled;
    
    const highlightStyle = document.getElementById('highlightStyle');
    if (highlightStyle) highlightStyle.value = this.settings.highlightStyle;
    
    const highlightColor = document.getElementById('highlightColor');
    if (highlightColor) highlightColor.value = this.settings.highlightColor;
    
    const textColor = document.getElementById('textColor');
    if (textColor) textColor.value = this.settings.textColor;
    
    const localServiceHost = document.getElementById('localServiceHost');
    if (localServiceHost) localServiceHost.value = this.settings.localServiceHost;
    
    const localServicePort = document.getElementById('localServicePort');
    if (localServicePort) localServicePort.value = this.settings.localServicePort;
    
    this.updateColorPreview();
    
    // 更新扩展图标
    this.updateExtensionIcon(this.settings.highlightEnabled);
  }

  // 保存高亮开关设置
  async saveHighlightEnabled() {
    try {
      const highlightEnabledEl = document.getElementById('highlightEnabled');
      if (!highlightEnabledEl) return;
      
      const highlightEnabled = highlightEnabledEl.checked;
      
      // 保存到存储
      await chrome.storage.local.set({ highlightEnabled: highlightEnabled });
      
      // 更新本地设置
      this.settings.highlightEnabled = highlightEnabled;
      
      console.log('WzSLinker Options: 高亮开关已更新', highlightEnabled);
      
      // 通知所有标签页更新设置
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'updateSettings',
            settings: { highlightEnabled: highlightEnabled }
          });
        } catch (error) {
          // 忽略无法发送消息的标签页
        }
      }
      
      // 更新扩展图标
      this.updateExtensionIcon(highlightEnabled);
      
    } catch (error) {
      console.error('保存高亮开关设置失败:', error);
    }
  }

  // 更新扩展图标
  async updateExtensionIcon(enabled) {
    try {
      // 使用badge来显示状态
      await chrome.action.setBadgeText({
        text: enabled ? '' : 'OFF'
      });
      
      await chrome.action.setBadgeBackgroundColor({
        color: enabled ? '#00cc66' : '#ff3333'
      });
      
      // 设置图标标题
      await chrome.action.setTitle({
        title: enabled ? 'WzSLinker - 股票联动工具（已启用）' : 'WzSLinker - 股票联动工具（已禁用）'
      });
      
      console.log('WzSLinker Options: 扩展图标已更新', { enabled });
      
    } catch (error) {
      console.error('更新扩展图标失败:', error);
    }
  }

  // 保存设置
  async saveSettings() {
    try {
      // 收集所有设置
      const highlightEnabledEl = document.getElementById('highlightEnabled');
      const highlightStyleEl = document.getElementById('highlightStyle');
      const highlightColorEl = document.getElementById('highlightColor');
      const textColorEl = document.getElementById('textColor');
      const localServiceHostEl = document.getElementById('localServiceHost');
      const localServicePortEl = document.getElementById('localServicePort');
      const requestTimeoutEl = document.getElementById('requestTimeout');
      const retryEnabledEl = document.getElementById('retryEnabled');
      const maxRetriesEl = document.getElementById('maxRetries');
      
      const newSettings = {
        highlightEnabled: highlightEnabledEl ? highlightEnabledEl.checked : this.settings.highlightEnabled,
        highlightStyle: highlightStyleEl ? highlightStyleEl.value : this.settings.highlightStyle,
        highlightColor: highlightColorEl ? highlightColorEl.value : this.settings.highlightColor,
        textColor: textColorEl ? textColorEl.value : this.settings.textColor,
        localServiceHost: localServiceHostEl ? localServiceHostEl.value : this.settings.localServiceHost,
        localServicePort: localServicePortEl ? localServicePortEl.value.trim() : this.settings.localServicePort,
        requestTimeout: requestTimeoutEl ? parseInt(requestTimeoutEl.value) : this.settings.requestTimeout,
        retryEnabled: retryEnabledEl ? retryEnabledEl.checked : this.settings.retryEnabled,
        maxRetries: maxRetriesEl ? parseInt(maxRetriesEl.value) : this.settings.maxRetries
      };

      // 验证端口号
      const port = newSettings.localServicePort;
      if (!/^\d+$/.test(port) || parseInt(port) < 1 || parseInt(port) > 65535) {
        this.showMessage('端口号必须是1-65535之间的数字', 'error');
        return;
      }

      // 验证超时时间
      if (newSettings.requestTimeout < 1000 || newSettings.requestTimeout > 30000) {
        this.showMessage('超时时间必须在1000-30000毫秒之间', 'error');
        return;
      }

      // 验证重试次数
      if (newSettings.maxRetries < 0 || newSettings.maxRetries > 5) {
        this.showMessage('重试次数必须在0-5之间', 'error');
        return;
      }

      console.log('WzSLinker Options: 保存设置', newSettings);

      // 保存到存储
      await chrome.storage.local.set(newSettings);
      
      // 更新本地设置
      Object.assign(this.settings, newSettings);
      
      this.showMessage('设置已保存！', 'success');
      
      // 通知所有标签页更新设置
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'updateSettings',
            settings: newSettings
          });
        } catch (error) {
          // 忽略无法发送消息的标签页
        }
      }
      
      // 更新扩展图标
      this.updateExtensionIcon(newSettings.highlightEnabled);
      
    } catch (error) {
      console.error('保存设置失败:', error);
      this.showMessage('保存设置失败: ' + error.message, 'error');
    }
  }

  // 导入股票数据
  importStockData() {
    document.getElementById('stockDataUpload').click();
  }

  // 处理文件上传
  handleFileUpload(file) {
    if (!file) return;
    
    const reader = new FileReader();
    reader.onload = async (e) => {
      try {
        const data = JSON.parse(e.target.result);
        
        // 验证数据格式
        if (typeof data !== 'object' || data === null) {
          throw new Error('无效的JSON格式');
        }
        
        // 更新股票数据
        this.settings.stockData = data;
        await chrome.storage.local.set({ stockData: data });
        
        this.showMessage(`成功导入 ${Object.keys(data).length} 条股票数据`, 'success');
        
        // 通知所有标签页重新加载数据
        const tabs = await chrome.tabs.query({});
        for (const tab of tabs) {
          try {
            await chrome.tabs.sendMessage(tab.id, {
              action: 'reloadStockData',
              data: data
            });
          } catch (error) {
            // 忽略无法发送消息的标签页
          }
        }
        
      } catch (error) {
        this.showMessage('导入失败: ' + error.message, 'error');
      }
    };
    reader.readAsText(file);
  }

  // 导出股票数据
  exportStockData() {
    const data = this.settings.stockData;
    const jsonString = JSON.stringify(data, null, 2);
    const blob = new Blob([jsonString], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    
    const a = document.createElement('a');
    a.href = url;
    a.download = `wzslinker-stock-data-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    this.showMessage('股票数据已导出', 'success');
  }

  // 添加单个股票
  async addSingleStock() {
    const nameInput = document.getElementById('addStockNameOpt');
    const codeInput = document.getElementById('addStockCodeOpt');
    
    const name = nameInput.value.trim();
    const code = codeInput.value.trim();
    
    if (!name) {
      this.showMessage('请输入股票名称', 'error');
      nameInput.focus();
      return;
    }
    
    if (!code || !/^\d{6}$/.test(code)) {
      this.showMessage('请输入6位数字股票代码', 'error');
      codeInput.focus();
      return;
    }
    
    try {
      // 添加到股票数据
      this.settings.stockData[name] = code;
      await chrome.storage.local.set({ stockData: this.settings.stockData });
      
      // 清空输入框
      nameInput.value = '';
      codeInput.value = '';
      
      this.showMessage(`成功添加股票: ${name} (${code})`, 'success');
      
      // 通知所有标签页重新加载数据并重新处理页面
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'reloadStockData',
            data: this.settings.stockData
          });
          // 重新处理页面
          await chrome.tabs.sendMessage(tab.id, {
            action: 'processPage'
          });
        } catch (error) {
          // 忽略无法发送消息的标签页
        }
      }
    } catch (error) {
      console.error('添加股票失败:', error);
      this.showMessage('添加股票失败: ' + error.message, 'error');
    }
  }

  // 清空所有股票数据
  async clearAllStockData() {
    if (confirm('确定要清空所有股票数据吗？此操作不可恢复！')) {
      this.settings.stockData = {};
      await chrome.storage.local.set({ stockData: {} });
      
      this.showMessage('所有股票数据已清空', 'success');
      
      // 通知所有标签页重新加载数据
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'reloadStockData',
            data: {}
          });
        } catch (error) {
          // 忽略无法发送消息的标签页
        }
      }
    }
  }

  // 清空联动记录
  async clearLinkedHistory() {
    if (confirm('确定要清空所有联动记录吗？')) {
      this.settings.linkHistory = [];
      await chrome.storage.local.set({ linkHistory: [] });
      this.showMessage('联动记录已清空', 'success');
    }
  }

  // 重置所有设置
  async resetSettings() {
    if (confirm('确定要重置所有设置为默认值吗？此操作不可恢复！')) {
      const defaultSettings = this.getDefaultSettings();
      
      await chrome.storage.local.clear();
      await chrome.storage.local.set(defaultSettings);
      
      this.settings = defaultSettings;
      this.updateUI();
      
      this.showMessage('所有设置已重置为默认值', 'success');
      
      // 通知所有标签页重新加载设置
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'resetSettings'
          });
        } catch (error) {
          // 忽略无法发送消息的标签页
        }
      }
    }
  }

  // 更新颜色预览
  updateColorPreview() {
    const highlightEnabled = document.getElementById('highlightEnabled');
    const highlightColor = document.getElementById('highlightColor');
    const textColor = document.getElementById('textColor');
    const highlightStyle = document.getElementById('highlightStyle');
    const preview = document.querySelector('#stylePreview .preview-stock');
    
    if (!preview || !highlightColor || !textColor || !highlightStyle) return;
    
    const enabled = highlightEnabled ? highlightEnabled.checked : true;
    const style = highlightStyle.value;
    const bgColor = highlightColor.value;
    const txtColor = textColor.value;
    
    // 重置样式
    preview.style.background = '';
    preview.style.color = '';
    preview.style.boxShadow = '';
    preview.style.border = '';
    preview.style.textDecoration = '';
    preview.style.borderBottom = '';
    preview.style.fontWeight = '';
    
    if (!enabled) {
      // 如果高亮被禁用，显示为普通文本
      preview.style.background = 'transparent';
      preview.style.color = 'inherit';
      preview.style.fontWeight = 'normal';
      return;
    }
    
    switch (style) {
      case 'normal':
        preview.style.background = bgColor;
        preview.style.color = txtColor;
        preview.style.fontWeight = 'bold';
        break;
        
      case 'glow':
        const rgb = this.hexToRgb(bgColor);
        preview.style.background = 'transparent';
        preview.style.color = txtColor;
        preview.style.boxShadow = `0 0 10px rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.8)`;
        preview.style.fontWeight = 'bold';
        break;
        
      case 'none':
        preview.style.background = 'transparent';
        preview.style.color = 'inherit';
        preview.style.textDecoration = 'underline dotted red';
        preview.style.borderBottom = '1px dotted red';
        preview.style.fontWeight = 'normal';
        break;
    }
  }

  // 十六进制颜色转RGB（使用公共工具函数）
  hexToRgb(hex) {
    return window.WzSLinkerUtils ? window.WzSLinkerUtils.hexToRgb(hex) : this._hexToRgb(hex);
  }

  // 备用十六进制颜色转RGB函数
  _hexToRgb(hex) {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? {
      r: parseInt(result[1], 16),
      g: parseInt(result[2], 16),
      b: parseInt(result[3], 16)
    } : { r: 255, g: 243, b: 205 }; // 默认颜色
  }

  // 测试连接
  async testConnection() {
    const button = document.getElementById('testConnectionButton');
    const hostEl = document.getElementById('localServiceHost');
    const portEl = document.getElementById('localServicePort');
    
    const host = hostEl?.value || '127.0.0.1';
    const port = portEl?.value || '3000';
    const url = `http://${host}:${port}`;
    
    if (button) {
      button.textContent = '测试中...';
      button.disabled = true;
    }
    
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 5000);
      
      // 发送一个真实的测试请求，而不是简单的test请求
      const response = await fetch(`${url}/tdx`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          code: '000001',
          name: '平安银行',
          url: 'test-connection'
        }),
        signal: controller.signal
      });
      
      clearTimeout(timeoutId);
      
      if (response.ok) {
        const result = await response.json();
        this.showMessage(`连接成功！Wzslinker本地服务运行正常 (${host}:${port})`, 'success');
        console.log('测试连接成功，服务响应:', result);
      } else {
        // 即使HTTP状态不是200，如果服务响应了，也说明服务在运行
        const errorText = await response.text();
        if (errorText) {
          this.showMessage(`连接成功！Wzslinker本地服务运行正常 (${host}:${port})`, 'success');
        } else {
          throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
      }
    } catch (error) {
      console.error('测试连接失败:', error);
      if (error.name === 'AbortError') {
        this.showMessage(`连接超时 (${host}:${port})，请检查Wzslinker本地服务是否正在运行`, 'error');
      } else if (error.message.includes('Failed to fetch') || error.message.includes('ERR_CONNECTION_REFUSED')) {
        this.showMessage(`连接失败 (${host}:${port})，请检查Wzslinker本地服务是否正在运行。访问官网：https://hrfocus.top`, 'error');
      } else {
        this.showMessage(`连接失败 (${host}:${port})，请检查Wzslinker本地服务是否正在运行。访问官网：https://hrfocus.top`, 'error');
      }
    } finally {
      if (button) {
        button.textContent = '测试连接';
        button.disabled = false;
      }
    }
  }

  // 显示消息（使用公共工具函数）
  showMessage(text, type = 'info') {
    if (window.WzSLinkerUtils) {
      window.WzSLinkerUtils.showMessage(text, type, 'statusMessage');
    } else {
      this._showMessage(text, type);
    }
  }

  // 备用显示消息函数
  _showMessage(text, type = 'info') {
    const messageEl = document.getElementById('statusMessage');
    if (messageEl) {
      messageEl.textContent = text;
      messageEl.className = `message ${type}`;
      messageEl.style.display = 'block';
      
      setTimeout(() => {
        messageEl.style.display = 'none';
      }, 3000);
    }
  }
}

// 初始化设置管理器
let optionsManager;
document.addEventListener('DOMContentLoaded', () => {
  optionsManager = new OptionsManager();
});