// WzSLinker 弹出窗口逻辑

class PopupManager {
  constructor() {
    this.init();
  }

  async init() {
    await this.loadSettings();
    await this.loadHistory();
    await this.loadStats();
    await this.loadUpdateInfo();
    this.setupEventListeners();
    this.updateUI();
    
    // 监听存储变化，同步状态
    chrome.storage.onChanged.addListener((changes, namespace) => {
      if (namespace === 'local') {
        if (changes.highlightEnabled || changes.highlightStyle || changes.highlightColor || changes.textColor) {
          this.loadSettings();
          this.updateUI();
        }
        if (changes.currentHighlightCount) {
          this.stats.highlightCount = changes.currentHighlightCount.newValue || 0;
          this.updateStats();
        }
      }
    });
  }

  // 加载设置
  async loadSettings() {
    try {
      const result = await chrome.storage.local.get([
        'highlightEnabled',
        'highlightStyle',
        'highlightColor',
        'textColor',
        'stockData',
        'showPriceEnabled',
        'domainBlocklistEnabled',
        'domainBlocklist',
        'domainAllowlistEnabled',
        'domainAllowlist'
      ]);
      
      const defaultSettings = window.WzSLinkerUtils ? window.WzSLinkerUtils.getDefaultSettings() : this._getDefaultSettings();
      this.settings = {
        highlightEnabled: result.highlightEnabled !== false,
        highlightStyle: result.highlightStyle || defaultSettings.highlightStyle,
        highlightColor: result.highlightColor || defaultSettings.highlightColor,
        textColor: result.textColor || defaultSettings.textColor,
        stockData: result.stockData || defaultSettings.stockData,
        showPriceEnabled: result.showPriceEnabled !== undefined ? result.showPriceEnabled : true,
        domainBlocklistEnabled: result.domainBlocklistEnabled || false,
        domainBlocklist: result.domainBlocklist || [],
        domainAllowlistEnabled: result.domainAllowlistEnabled || false,
        domainAllowlist: result.domainAllowlist || []
      };
    } catch (error) {
      console.error('加载设置失败:', error);
      this.settings = window.WzSLinkerUtils ? window.WzSLinkerUtils.getDefaultSettings() : this._getDefaultSettings();
    }
  }

  // 加载历史记录
  async loadHistory() {
    try {
      const result = await chrome.storage.local.get(['linkHistory']);
      this.history = result.linkHistory || [];
    } catch (error) {
      console.error('加载历史记录失败:', error);
      this.history = [];
    }
  }

  // 加载统计信息
  async loadStats() {
    try {
      const result = await chrome.storage.local.get(['stats', 'currentHighlightCount']);
      this.stats = result.stats || {
        highlightCount: result.currentHighlightCount || 0,
        stockCount: 0,
        historyCount: 0
      };
      // 使用实时的高亮计数
      this.stats.highlightCount = result.currentHighlightCount || 0;
    } catch (error) {
      console.error('加载统计信息失败:', error);
      this.stats = {
        highlightCount: 0,
        stockCount: 0,
        historyCount: 0
      };
    }
  }

  // 加载更新信息
  async loadUpdateInfo() {
    try {
      const result = await chrome.storage.local.get(['pendingUpdate']);
      this.updateInfo = result.pendingUpdate || null;
    } catch (error) {
      console.error('加载更新信息失败:', error);
      this.updateInfo = null;
    }
  }

  // 设置事件监听器
  setupEventListeners() {
    // 高亮开关
    const highlightToggle = document.getElementById('highlightToggle');
    if (highlightToggle) {
      highlightToggle.addEventListener('change', (e) => {
        this.toggleHighlight(e.target.checked);
      });
    }

    // 高亮样式选择
    const highlightStyle = document.getElementById('highlightStyle');
    if (highlightStyle) {
      highlightStyle.addEventListener('change', (e) => {
        this.changeHighlightStyle(e.target.value);
      });
    }

    // 涨跌幅显示开关
    const priceDisplayToggle = document.getElementById('priceDisplayToggle');
    if (priceDisplayToggle) {
      priceDisplayToggle.addEventListener('change', (e) => {
        this.togglePriceDisplay(e.target.checked);
      });
    }

    // 清空历史
    const clearHistory = document.getElementById('clearHistory');
    if (clearHistory) {
      clearHistory.addEventListener('click', () => {
        this.clearHistory();
      });
    }

    // 导入JSON
    const importJson = document.getElementById('importJson');
    if (importJson) {
      importJson.addEventListener('click', () => {
        const importFile = document.getElementById('importFile');
        if (importFile) {
          importFile.click();
        }
      });
    }

    // 文件选择
    const importFile = document.createElement('input');
    importFile.type = 'file';
    importFile.accept = '.json';
    importFile.id = 'importFile';
    importFile.style.display = 'none';
    importFile.addEventListener('change', (e) => {
      this.importJsonFile(e.target.files[0]);
    });
    document.body.appendChild(importFile);

    // 导出JSON
    const exportJson = document.getElementById('exportJson');
    if (exportJson) {
      exportJson.addEventListener('click', () => {
        this.exportJsonFile();
      });
    }

    // 添加股票
    const addBtn = document.getElementById('addBtn');
    if (addBtn) {
      addBtn.addEventListener('click', () => {
        this.addSingleStock();
      });
    }

    // 设置按钮
    const settingsButton = document.getElementById('settingsButton');
    if (settingsButton) {
      settingsButton.addEventListener('click', () => {
        chrome.runtime.openOptionsPage();
      });
    }

    // 关闭按钮
    const closeButton = document.getElementById('closeButton');
    if (closeButton) {
      closeButton.addEventListener('click', () => {
        window.close();
      });
    }

    // 回车键添加股票
    const stockCode = document.getElementById('stockCode');
    if (stockCode) {
      stockCode.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
          this.addSingleStock();
        }
      });
    }

    // 复制微信号按钮
    const copyBtn = document.querySelector('.copy-btn');
    if (copyBtn) {
      copyBtn.addEventListener('click', () => {
        this.copyWechatNumber();
      });
    }

    // 域名管理 - 屏蔽模式开关
    const blocklistToggle = document.getElementById('blocklistToggle');
    if (blocklistToggle) {
      blocklistToggle.addEventListener('change', (e) => {
        this.toggleBlocklist(e.target.checked);
      });
    }

    // 域名管理 - 允许模式开关
    const allowlistToggle = document.getElementById('allowlistToggle');
    if (allowlistToggle) {
      allowlistToggle.addEventListener('change', (e) => {
        this.toggleAllowlist(e.target.checked);
      });
    }

    // 域名管理 - 保存按钮
    const saveDomainSettings = document.getElementById('saveDomainSettings');
    if (saveDomainSettings) {
      saveDomainSettings.addEventListener('click', () => {
        this.saveDomainSettings();
      });
    }

    // 域名管理 - 清空按钮
    const clearDomainSettings = document.getElementById('clearDomainSettings');
    if (clearDomainSettings) {
      clearDomainSettings.addEventListener('click', () => {
        this.clearDomainSettings();
      });
    }
  }

  // 更新UI
  updateUI() {
    // 更新高亮开关
    const highlightToggle = document.getElementById('highlightToggle');
    if (highlightToggle) {
      highlightToggle.checked = this.settings.highlightEnabled;
    }
    
    // 更新高亮样式
    const highlightStyle = document.getElementById('highlightStyle');
    if (highlightStyle) {
      highlightStyle.value = this.settings.highlightStyle;
    }
    
    // 更新涨跌幅显示开关
    const priceDisplayToggle = document.getElementById('priceDisplayToggle');
    if (priceDisplayToggle) {
      priceDisplayToggle.checked = this.settings.showPriceEnabled;
    }
    
    // 更新涨跌幅开关标签
    const priceToggleLabel = document.getElementById('priceToggleLabel');
    if (priceToggleLabel) {
      priceToggleLabel.textContent = this.settings.showPriceEnabled ? '已启用' : '已禁用';
      priceToggleLabel.style.color = this.settings.showPriceEnabled ? '#00cc66' : '#999999';
    }
    
    // 更新域名管理UI
    this.updateDomainUI();
    
    // 更新历史记录
    this.updateHistoryList();
    
    // 更新统计信息
    this.updateStats();
    
    // 更新版本信息
    this.updateVersionInfo();
  }

  // 更新历史记录列表
  updateHistoryList() {
    const historyList = document.getElementById('historyList');
    const emptyHistory = document.getElementById('emptyHistory');
    
    if (this.history.length === 0) {
      historyList.style.display = 'none';
      emptyHistory.style.display = 'block';
      return;
    }
    
    historyList.style.display = 'block';
    emptyHistory.style.display = 'none';
    
    historyList.innerHTML = this.history.slice(0, 30).map(item => `
      <div class="history-item" data-item-id="${item.id}">
        <div class="history-content">
          <div class="history-stock">${this.escapeHtml(item.name)} (${item.code})</div>
          <div class="history-details">
            <span>${item.time}</span>
            <span>${this.getDomainFromUrl(item.url)}</span>
          </div>
        </div>
        <div class="history-actions">
          <button class="success relink-btn" data-code="${item.code}" data-name="${this.escapeHtml(item.name)}">再次联动</button>
          <button class="danger delete-btn" data-id="${item.id}">删除</button>
        </div>
      </div>
    `).join('');
    
    // 为所有"再次联动"按钮绑定事件
    historyList.querySelectorAll('.relink-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const code = e.target.getAttribute('data-code');
        const name = e.target.getAttribute('data-name');
        this.relinkStock(code, name);
      });
    });
    
    // 为所有"删除"按钮绑定事件
    historyList.querySelectorAll('.delete-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const id = e.target.getAttribute('data-id');
        this.deleteHistoryItem(id);
      });
    });
  }
  
  // HTML转义函数，防止XSS和语法错误
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // 更新统计信息
  updateStats() {
    document.getElementById('highlightCount').textContent = this.stats.highlightCount;
    document.getElementById('stockCount').textContent = Object.keys(this.settings.stockData).length;
    document.getElementById('historyCount').textContent = this.history.length;
  }

  // 切换高亮
  async toggleHighlight(enabled) {
    this.settings.highlightEnabled = enabled;
    await chrome.storage.local.set({ highlightEnabled: enabled });
    
    // 通知所有标签页
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      try {
        await chrome.tabs.sendMessage(tab.id, {
          action: 'toggleHighlight',
          enabled: enabled
        });
      } catch (error) {
        // 忽略无法发送消息的标签页
      }
    }
    
    // 通知background script更新图标状态
    chrome.runtime.sendMessage({
      action: 'updateIconState',
      enabled: enabled
    });
    
    this.showMessage(enabled ? '高亮已开启' : '高亮已关闭', 'success');
  }

  // 改变高亮样式
  async changeHighlightStyle(style) {
    this.settings.highlightStyle = style;
    await chrome.storage.local.set({ highlightStyle: style });
    
    // 通知所有标签页
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      try {
        await chrome.tabs.sendMessage(tab.id, {
          action: 'updateSettings',
          settings: { highlightStyle: style }
        });
      } catch (error) {
        // 忽略无法发送消息的标签页
      }
    }
    
    this.showMessage(`高亮样式已切换为: ${this.getStyleName(style)}`, 'success');
  }

  // 切换涨跌幅显示
  async togglePriceDisplay(enabled) {
    this.settings.showPriceEnabled = enabled;
    await chrome.storage.local.set({ showPriceEnabled: enabled });
    
    // 更新标签文本和颜色
    const priceToggleLabel = document.getElementById('priceToggleLabel');
    if (priceToggleLabel) {
      priceToggleLabel.textContent = enabled ? '已启用' : '已禁用';
      priceToggleLabel.style.color = enabled ? '#00cc66' : '#999999';
    }
    
    // 通知所有标签页
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      try {
        await chrome.tabs.sendMessage(tab.id, {
          action: 'updateSettings',
          settings: { showPriceEnabled: enabled }
        });
      } catch (error) {
        // 忽略无法发送消息的标签页
      }
    }
    
    this.showMessage(enabled ? '涨跌幅显示已开启' : '涨跌幅显示已关闭', 'success');
  }

  // 清空历史记录
  async clearHistory() {
    if (confirm('确定要清空所有联动记录吗？')) {
      this.history = [];
      await chrome.storage.local.set({ linkHistory: [] });
      this.updateHistoryList();
      this.updateStats();
      this.showMessage('历史记录已清空', 'success');
    }
  }

  // 导入JSON文件
  importJsonFile(file) {
    if (!file) return;
    
    const reader = new FileReader();
    reader.onload = async (e) => {
      try {
        const data = JSON.parse(e.target.result);
        
        // 验证数据格式
        if (typeof data !== 'object' || data === null) {
          throw new Error('无效的JSON格式');
        }
        
        // 更新股票数据
        this.settings.stockData = data;
        await chrome.storage.local.set({ stockData: data });
        
        // 通知content script重新加载数据
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) {
          chrome.tabs.sendMessage(tab.id, {
            action: 'reloadStockData',
            data: data
          });
        }
        
        this.updateStats();
        this.showMessage(`成功导入 ${Object.keys(data).length} 条股票数据`, 'success');
        
      } catch (error) {
        this.showMessage('导入失败: ' + error.message, 'error');
      }
    };
    reader.readAsText(file);
  }

  // 导出JSON文件
  exportJsonFile() {
    const data = this.settings.stockData;
    const jsonString = JSON.stringify(data, null, 2);
    const blob = new Blob([jsonString], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    
    const a = document.createElement('a');
    a.href = url;
    a.download = `wzslinker-stock-data-${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    this.showMessage('股票数据已导出', 'success');
  }

  // 添加单个股票
  async addSingleStock() {
    const nameInput = document.getElementById('stockName');
    const codeInput = document.getElementById('stockCode');
    
    const name = nameInput.value.trim();
    const code = codeInput.value.trim();
    
    if (!name) {
      this.showMessage('请输入股票名称', 'error');
      nameInput.focus();
      return;
    }
    
    if (!code || !/^\d{6}$/.test(code)) {
      this.showMessage('请输入6位数字股票代码', 'error');
      codeInput.focus();
      return;
    }
    
    try {
      // 添加到股票数据
      this.settings.stockData[name] = code;
      await chrome.storage.local.set({ stockData: this.settings.stockData });
      
      // 通知content script重新加载数据
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'reloadStockData',
            data: this.settings.stockData
          });
        } catch (error) {
          console.log('无法发送消息到content script:', error);
        }
      }
      
      // 清空输入框
      nameInput.value = '';
      codeInput.value = '';
      
      // 更新统计信息
      this.updateStats();
      this.showMessage(`成功添加股票: ${name} (${code})`, 'success');
      
      // 重新处理页面以显示新添加的股票
      if (tab) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'processPage'
          });
        } catch (error) {
          console.log('无法触发页面重新处理:', error);
        }
      }
      
    } catch (error) {
      console.error('添加股票失败:', error);
      this.showMessage('添加股票失败: ' + error.message, 'error');
    }
  }

  // 再次联动股票
  async relinkStock(code, name) {
    console.log('WzSLinker Popup: 开始再次联动', { code, name });
    
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      console.log('WzSLinker Popup: 获取到当前标签页', { tabId: tab?.id, url: tab?.url });
      
      if (tab) {
        // 检查content script是否已加载
        console.log('WzSLinker Popup: 检查content script状态');
        
        try {
          // 先发送一个ping消息测试content script是否响应
          const pingResponse = await chrome.tabs.sendMessage(tab.id, { action: 'ping' });
          console.log('WzSLinker Popup: Content script响应', pingResponse);
        } catch (pingError) {
          console.error('WzSLinker Popup: Content script未响应', pingError);
          this.showMessage('页面未加载完成，请刷新页面后重试', 'error');
          return;
        }
        
        // 发送联动请求到content script
        console.log('WzSLinker Popup: 发送联动消息到content script', { code, name });
        
        const response = await chrome.tabs.sendMessage(tab.id, {
          action: 'linkStock',
          code: code,
          name: name
        });
        
        console.log('WzSLinker Popup: 收到content script响应', response);
        
        // 检查响应是否成功
        if (response && response.success) {
          // 显示联动成功消息
          this.showMessage(`已联动: ${name} (${code})`, 'success');
          
          // 更新历史记录（添加新的联动记录）
          await this.addToHistory(code, name, tab.url);
          
          console.log('WzSLinker Popup: 再次联动完成', { code, name });
        } else {
          // 联动失败
          const errorMsg = response?.error || '未知错误';
          console.error('WzSLinker Popup: 联动失败', errorMsg);
          this.showMessage(`联动失败: ${errorMsg}`, 'error');
        }
        
      } else {
        console.error('WzSLinker Popup: 无法获取当前标签页');
        this.showMessage('无法获取当前标签页', 'error');
      }
    } catch (error) {
      console.error('WzSLinker Popup: 再次联动失败', error);
      
      // 根据错误类型显示不同的错误信息
      if (error.message.includes('Could not establish connection')) {
        this.showMessage('页面未加载完成，请刷新页面后重试', 'error');
      } else if (error.message.includes('Receiving end does not exist')) {
        this.showMessage('扩展未在页面中加载，请刷新页面', 'error');
      } else {
        this.showMessage('联动失败，请检查Wzslinker本地服务', 'error');
      }
    }
  }

  // 添加到历史记录
  async addToHistory(code, name, url) {
    try {
      const historyItem = {
        id: Date.now().toString(),
        code: code,
        name: name,
        time: new Date().toLocaleString(),
        url: url || 'popup'
      };
      
      // 添加到历史记录开头
      this.history.unshift(historyItem);
      
      // 保留最近30条记录
      this.history = this.history.slice(0, 30);
      
      // 保存到存储
      await chrome.storage.local.set({ linkHistory: this.history });
      
      // 更新UI
      this.updateHistoryList();
      this.updateStats();
      
    } catch (error) {
      console.error('添加历史记录失败:', error);
    }
  }

  // 删除历史记录项
  async deleteHistoryItem(id) {
    this.history = this.history.filter(item => item.id !== id);
    await chrome.storage.local.set({ linkHistory: this.history });
    this.updateHistoryList();
    this.updateStats();
    this.showMessage('记录已删除', 'success');
  }

  // 显示消息
  showMessage(text, type = 'info') {
    const messageEl = document.getElementById('statusMessage');
    messageEl.textContent = text;
    messageEl.className = `message ${type}`;
    messageEl.style.display = 'block';
    
    setTimeout(() => {
      messageEl.style.display = 'none';
    }, 3000);
  }

  // 获取样式名称
  getStyleName(style) {
    const names = {
      'normal': '普通',
      'glow': '发光',
      'none': '无高亮'
    };
    return names[style] || style;
  }

  // 从URL获取域名
  getDomainFromUrl(url) {
    try {
      return new URL(url).hostname;
    } catch {
      return '未知';
    }
  }


  // 清空历史记录
  async clearHistory() {
    if (confirm('确定要清空所有历史记录吗？')) {
      this.history = [];
      await chrome.storage.local.set({ linkHistory: [] });
      this.updateHistoryList();
      this.updateStats();
      this.showMessage('历史记录已清空', 'success');
    }
  }

  // 复制微信号
  async copyWechatNumber() {
    const wechatNumber = '370589873';
    try {
      await navigator.clipboard.writeText(wechatNumber);
      this.showMessage('微信号已复制到剪贴板', 'success');
      
      // 更新按钮状态
      const copyBtn = document.querySelector('.copy-btn');
      if (copyBtn) {
        const originalText = copyBtn.innerHTML;
        copyBtn.innerHTML = '✅';
        copyBtn.style.background = '#00cc66';
        
        setTimeout(() => {
          copyBtn.innerHTML = originalText;
          copyBtn.style.background = '#4a90e2';
        }, 2000);
      }
    } catch (error) {
      console.error('复制失败:', error);
      this.showMessage('复制失败，请手动复制微信号', 'error');
    }
  }

  // 更新版本信息
  updateVersionInfo() {
    const versionInfo = document.getElementById('versionInfo');
    if (versionInfo) {
      const currentVersion = chrome.runtime.getManifest().version;
      let versionHtml = `<span class="current-version">当前版本: ${currentVersion}</span>`;
      
      if (this.updateInfo && this.updateInfo.version) {
        versionHtml += `<span class="update-badge" id="updateBadge">🎉 v${this.updateInfo.version} 可更新</span>`;
      }
      
      versionInfo.innerHTML = versionHtml;
      
      // 添加点击事件监听
      const updateBadge = document.getElementById('updateBadge');
      if (updateBadge) {
        updateBadge.addEventListener('click', () => {
          this.showUpdateInfo();
        });
      }
    }
  }

  // 显示更新信息
  showUpdateInfo() {
    if (this.updateInfo) {
      if (confirm(`发现新版本 ${this.updateInfo.version}！\n\n是否立即前往下载页面？`)) {
        window.open(this.updateInfo.downloadUrl || 'https://hrfocus.top/extension-download', '_blank');
      }
    }
  }

  // ==================== 域名管理功能 ====================

  /**
   * 智能解析域名（支持多种格式，包括本地文件）
   * @param {string} input - 用户输入的域名/URL/文件路径
   * @returns {string} 解析后的域名或文件路径
   */
  parseDomain(input) {
    if (!input || typeof input !== 'string') return '';
    
    input = input.trim();
    if (!input) return '';
    
    try {
      // 如果是 file:// 协议，返回完整的文件路径（标准化）
      if (/^file:\/\//i.test(input)) {
        // 标准化文件路径
        let filePath = input.replace(/^file:\/\//i, 'file:///');
        // 统一使用正斜杠
        filePath = filePath.replace(/\\/g, '/');
        // URL解码（处理中文等字符）
        try {
          const urlObj = new URL(filePath);
          return urlObj.href;
        } catch (e) {
          return filePath;
        }
      }
      
      // 如果是 IP:端口 格式，直接返回
      if (/^\d+\.\d+\.\d+\.\d+(:\d+)?$/.test(input)) {
        return input;
      }
      
      // 如果没有协议，添加 https://
      let url = input;
      if (!/^https?:\/\//i.test(input)) {
        url = 'https://' + input;
      }
      
      // 解析 URL
      const urlObj = new URL(url);
      let hostname = urlObj.hostname;
      
      // 如果有端口，保留端口
      if (urlObj.port) {
        hostname += ':' + urlObj.port;
      }
      
      return hostname;
    } catch (error) {
      // 如果解析失败，尝试作为纯域名处理
      // 移除协议和路径
      let domain = input.replace(/^https?:\/\//i, '').split('/')[0];
      return domain.trim();
    }
  }

  /**
   * 解析域名列表
   * @param {string} text - 多行域名文本
   * @returns {Array<string>} 解析后的域名数组
   */
  parseDomainList(text) {
    if (!text || typeof text !== 'string') return [];
    
    const lines = text.split('\n');
    const domains = [];
    
    for (const line of lines) {
      const parsed = this.parseDomain(line);
      if (parsed && !domains.includes(parsed)) {
        domains.push(parsed);
      }
    }
    
    return domains;
  }

  /**
   * 更新域名管理UI
   */
  updateDomainUI() {
    // 更新屏蔽模式开关
    const blocklistToggle = document.getElementById('blocklistToggle');
    const blocklistContainer = document.getElementById('blocklistContainer');
    const blocklistLabel = document.getElementById('blocklistLabel');
    
    if (blocklistToggle) {
      blocklistToggle.checked = this.settings.domainBlocklistEnabled;
      if (blocklistContainer) {
        blocklistContainer.style.display = this.settings.domainBlocklistEnabled ? 'block' : 'none';
      }
      if (blocklistLabel) {
        blocklistLabel.style.color = this.settings.domainBlocklistEnabled ? '#00cc66' : '#666666';
      }
    }
    
    // 更新允许模式开关
    const allowlistToggle = document.getElementById('allowlistToggle');
    const allowlistContainer = document.getElementById('allowlistContainer');
    const allowlistLabel = document.getElementById('allowlistLabel');
    
    if (allowlistToggle) {
      allowlistToggle.checked = this.settings.domainAllowlistEnabled;
      if (allowlistContainer) {
        allowlistContainer.style.display = this.settings.domainAllowlistEnabled ? 'block' : 'none';
      }
      if (allowlistLabel) {
        allowlistLabel.style.color = this.settings.domainAllowlistEnabled ? '#00cc66' : '#666666';
      }
    }
    
    // 更新域名列表
    const blocklistInput = document.getElementById('blocklistInput');
    if (blocklistInput && this.settings.domainBlocklist) {
      blocklistInput.value = this.settings.domainBlocklist.join('\n');
    }
    
    const allowlistInput = document.getElementById('allowlistInput');
    if (allowlistInput && this.settings.domainAllowlist) {
      allowlistInput.value = this.settings.domainAllowlist.join('\n');
    }
  }

  /**
   * 切换屏蔽模式
   */
  async toggleBlocklist(enabled) {
    // 如果启用屏蔽模式，自动关闭允许模式
    if (enabled && this.settings.domainAllowlistEnabled) {
      this.settings.domainAllowlistEnabled = false;
      const allowlistToggle = document.getElementById('allowlistToggle');
      if (allowlistToggle) {
        allowlistToggle.checked = false;
      }
      const allowlistContainer = document.getElementById('allowlistContainer');
      if (allowlistContainer) {
        allowlistContainer.style.display = 'none';
      }
      await chrome.storage.local.set({ domainAllowlistEnabled: false });
    }
    
    this.settings.domainBlocklistEnabled = enabled;
    await chrome.storage.local.set({ domainBlocklistEnabled: enabled });
    
    const blocklistContainer = document.getElementById('blocklistContainer');
    const blocklistLabel = document.getElementById('blocklistLabel');
    if (blocklistContainer) {
      blocklistContainer.style.display = enabled ? 'block' : 'none';
    }
    if (blocklistLabel) {
      blocklistLabel.style.color = enabled ? '#00cc66' : '#666666';
    }
    
    // 通知所有content-script更新
    this.notifyDomainSettingsChanged();
    
    this.showMessage(enabled ? '已启用屏蔽模式' : '已关闭屏蔽模式', 'success');
  }

  /**
   * 切换允许模式
   */
  async toggleAllowlist(enabled) {
    // 如果启用允许模式，自动关闭屏蔽模式
    if (enabled && this.settings.domainBlocklistEnabled) {
      this.settings.domainBlocklistEnabled = false;
      const blocklistToggle = document.getElementById('blocklistToggle');
      if (blocklistToggle) {
        blocklistToggle.checked = false;
      }
      const blocklistContainer = document.getElementById('blocklistContainer');
      if (blocklistContainer) {
        blocklistContainer.style.display = 'none';
      }
      await chrome.storage.local.set({ domainBlocklistEnabled: false });
    }
    
    this.settings.domainAllowlistEnabled = enabled;
    await chrome.storage.local.set({ domainAllowlistEnabled: enabled });
    
    const allowlistContainer = document.getElementById('allowlistContainer');
    const allowlistLabel = document.getElementById('allowlistLabel');
    if (allowlistContainer) {
      allowlistContainer.style.display = enabled ? 'block' : 'none';
    }
    if (allowlistLabel) {
      allowlistLabel.style.color = enabled ? '#00cc66' : '#666666';
    }
    
    // 通知所有content-script更新
    this.notifyDomainSettingsChanged();
    
    this.showMessage(enabled ? '已启用允许模式' : '已关闭允许模式', 'success');
  }

  /**
   * 保存域名设置
   */
  async saveDomainSettings() {
    const blocklistInput = document.getElementById('blocklistInput');
    const allowlistInput = document.getElementById('allowlistInput');
    
    // 解析并保存屏蔽列表
    if (blocklistInput) {
      const blocklistText = blocklistInput.value;
      const parsedBlocklist = this.parseDomainList(blocklistText);
      this.settings.domainBlocklist = parsedBlocklist;
      await chrome.storage.local.set({ domainBlocklist: parsedBlocklist });
      
      // 更新显示（格式化）
      blocklistInput.value = parsedBlocklist.join('\n');
    }
    
    // 解析并保存允许列表
    if (allowlistInput) {
      const allowlistText = allowlistInput.value;
      const parsedAllowlist = this.parseDomainList(allowlistText);
      this.settings.domainAllowlist = parsedAllowlist;
      await chrome.storage.local.set({ domainAllowlist: parsedAllowlist });
      
      // 更新显示（格式化）
      allowlistInput.value = parsedAllowlist.join('\n');
    }
    
    // 通知所有content-script更新
    this.notifyDomainSettingsChanged();
    
    this.showMessage('域名设置已保存', 'success');
  }

  /**
   * 清空域名设置
   */
  async clearDomainSettings() {
    if (!confirm('确定要清空所有域名设置吗？此操作不可恢复。')) {
      return;
    }
    
    this.settings.domainBlocklist = [];
    this.settings.domainAllowlist = [];
    this.settings.domainBlocklistEnabled = false;
    this.settings.domainAllowlistEnabled = false;
    
    await chrome.storage.local.set({
      domainBlocklist: [],
      domainAllowlist: [],
      domainBlocklistEnabled: false,
      domainAllowlistEnabled: false
    });
    
    // 更新UI
    this.updateDomainUI();
    
    // 通知所有content-script更新
    this.notifyDomainSettingsChanged();
    
    this.showMessage('域名设置已清空', 'success');
  }

  /**
   * 通知所有content-script域名设置已更改
   */
  async notifyDomainSettingsChanged() {
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      try {
        await chrome.tabs.sendMessage(tab.id, {
          action: 'updateDomainSettings',
          settings: {
            domainBlocklistEnabled: this.settings.domainBlocklistEnabled,
            domainBlocklist: this.settings.domainBlocklist,
            domainAllowlistEnabled: this.settings.domainAllowlistEnabled,
            domainAllowlist: this.settings.domainAllowlist
          }
        });
      } catch (error) {
        // 忽略无法发送消息的标签页
      }
    }
  }

  // 备用默认设置函数
  _getDefaultSettings() {
    return {
      highlightStyle: 'none',
      highlightColor: '#fff3cd',
      textColor: '#ff0000',
      highlightEnabled: true,
      stockData: {}
    };
  }
}

// 初始化弹出窗口管理器
let popupManager;
document.addEventListener('DOMContentLoaded', () => {
  popupManager = new PopupManager();
});