// WzSLinker 公共工具函数

/**
 * 十六进制颜色转RGB
 * @param {string} hex - 十六进制颜色值
 * @returns {Object} RGB对象 {r, g, b}
 */
function hexToRgb(hex) {
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? {
    r: parseInt(result[1], 16),
    g: parseInt(result[2], 16),
    b: parseInt(result[3], 16)
  } : { r: 255, g: 243, b: 205 }; // 默认颜色
}

/**
 * 获取默认设置
 * @returns {Object} 默认设置对象
 */
function getDefaultSettings() {
  return {
    highlightStyle: 'none',
    highlightColor: '#fff3cd',
    textColor: '#ff0000',
    highlightEnabled: true,
    stockData: {},
    localServiceUrl: 'http://localhost:3000',
    linkHistory: []
  };
}

/**
 * 验证股票代码格式
 * @param {string} code - 股票代码
 * @returns {boolean} 是否为有效的6位数字代码
 */
function isValidStockCode(code) {
  return /^\d{6}$/.test(code);
}

/**
 * 显示消息（通用函数）
 * @param {string} text - 消息文本
 * @param {string} type - 消息类型 (success, error, info)
 * @param {string} elementId - 消息元素ID
 */
function showMessage(text, type = 'info', elementId = 'statusMessage') {
  const messageEl = document.getElementById(elementId);
  if (messageEl) {
    messageEl.textContent = text;
    messageEl.className = `message ${type}`;
    messageEl.style.display = 'block';
    
    setTimeout(() => {
      messageEl.style.display = 'none';
    }, 3000);
  }
}

// 如果在浏览器环境中，将函数暴露到全局
if (typeof window !== 'undefined') {
  window.WzSLinkerUtils = {
    hexToRgb,
    getDefaultSettings,
    isValidStockCode,
    showMessage
  };
}

// 如果在Node.js环境中，导出模块
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    hexToRgb,
    getDefaultSettings,
    isValidStockCode,
    showMessage
  };
}
